package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Set;

import org.junit.Test;

import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.MetricNamesRequest;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class MetricNamesHandlerTest extends HandlerTestBase {

    @Test
    public void metricNamesEmpty() {
        assertEquals(Set.of(), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(Set.of(), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate=abc"));
        assertEquals(Set.of(), metricNames(SHARD_COMPUTE, ""));
        assertEquals(Set.of(), metricNames(SHARD_CERTIFICATE_MANAGER, "host=*"));
    }

    @Test
    public void metricNamesMetricWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage, host=cluster");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit, host=cluster");
        addMetrics(shard, usage, limit);
        assertEquals(Set.of("quota.certificates_count.usage"), metricNames(SHARD_CERTIFICATE_MANAGER, "quota.certificates_count.usage{}"));
        assertEquals(Set.of("quota.certificates_count.usage"), metricNames(SHARD_CERTIFICATE_MANAGER, "'quota.certificates_count.us*'{}"));
        assertEquals(Set.of("quota.certificates_count.limit"), metricNames(SHARD_CERTIFICATE_MANAGER, "'quota.certificates_count.l*'{}"));
        assertEquals(Set.of("quota.certificates_count.usage", "quota.certificates_count.limit"), metricNames(SHARD_CERTIFICATE_MANAGER, ""));
    }

    @Test
    public void metrcNamesWhenResourceNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);

        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=fd3cdv65706qam9l4cob}"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "{certificate=fd3cdv65706qam9l4cob}"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=*}"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{}"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "{}"));
        assertEquals(Set.of(), metricNames(SHARD_CERTIFICATE_MANAGER, "{certificate!=fd3cdv65706qam9l4cob}"));
    }

    @Test
    public void metrcNamesByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate=my"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate=*"));
        assertEquals(Set.of("certificate.days_until_expiration"), metricNames(SHARD_CERTIFICATE_MANAGER, "certificate.days_until_expiration{certificate=fd3cdv65706qam9l4cob}"));
    }

    @Test
    public void metrcNamesMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "{}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "disk_write_latency{resource_id=solomon-stockpile-sas-01}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "disk_write_latency{resource_id=*, device=hdd-01}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "disk_write_latency{device=hdd-01}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "disk_write_latency{resource_id=*, device=epdplhd148habqj0np92}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "disk_write_latency{device=epdplhd148habqj0np92}"));
    }

    @Test // TODO: decide what todo with negative match by name or id, maybe need limit match by resource_id to only exact match (gordiychuk@)
    public void metrcNamesByNotName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_read_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");
        assertEquals(Set.of("disk_read_latency"), metricNames(SHARD_COMPUTE, "{device!=hdd-01}"));
        assertEquals(Set.of("disk_read_latency"), metricNames(SHARD_COMPUTE, "{resource_id=*, device!=hdd-01}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "{device!=hdd-16}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "{resource_id!=solomon-stockpile-sas-16, device!=hdd-16}"));
        assertEquals(Set.of("disk_write_latency"), metricNames(SHARD_COMPUTE, "{resource_id!=solomon-stockpile-sas-16}"));
        assertEquals(Set.of("disk_read_latency"), metricNames(SHARD_COMPUTE, "{resource_id!=solomon-stockpile-sas-0*}"));
    }

    private Set<String> metricNames(int numId, String selectors) {
        var handler = new MetricNamesHandler(shardResolver, referenceResolver, resourceFinder);
        var response = handler.metricNames(MetricNamesRequest.newBuilder()
                .setShardId(numId)
                .setSelectors(LabelSelectorConverter.selectorsToNewProto(SelectorsFormat.parse(selectors)))
                .build())
                .join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());
        return Set.copyOf(response.getNamesList());
    }
}
