package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Set;
import java.util.concurrent.ForkJoinPool;

import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.coremon.meta.db.memory.InMemoryMetricsDaoFactory;
import ru.yandex.solomon.coremon.meta.service.MetabaseShardResolverStub;
import ru.yandex.solomon.coremon.meta.service.ShardKeyAndMetricKey;
import ru.yandex.solomon.coremon.meta.service.cloud.NameResolverResourceFinder;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceLabel;
import ru.yandex.solomon.coremon.meta.service.cloud.ReferenceResolverStub;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.ResolveOneRequest;
import ru.yandex.solomon.metrics.client.StockpileClientStub;
import ru.yandex.solomon.name.resolver.client.NameResolverClientStub;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class ResolveOneHandlerTest extends HandlerTestBase {

    @Before
    public void setUp() {
        var stockpileClient = new StockpileClientStub(ForkJoinPool.commonPool());
        var metricDao = new InMemoryMetricsDaoFactory();

        shardResolver = new MetabaseShardResolverStub(SHARDS, metricDao, stockpileClient);
        shardResolver.awaitShardsReady();
        nameResolverClient = new NameResolverClientStub();
        referenceResolver = new ReferenceResolverStub(shardResolver);
        referenceResolver.addReference("compute", new ReferenceLabel("resource_id", Set.of("vm"), Set.of("compute"), Set.of("pre")));
        referenceResolver.addReference("compute", new ReferenceLabel("device", Set.of("disk"),  Set.of("compute"), Set.of("pre")));
        referenceResolver.addReference("certificate-manager", new ReferenceLabel("certificate", Set.of("certificate"),  Set.of("certificate-manager"), Set.of("prod")));
        resourceFinder = new NameResolverResourceFinder(nameResolverClient);
    }

    @After
    public void tearDown() throws Exception {
        if (shardResolver != null) {
            shardResolver.close();
        }
    }

    @Test
    public void resolveEmpty() {
        var shardLabels = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER).getShardKey().toLabels();
        var handler = new ResolveOneHandler(shardResolver, referenceResolver, resourceFinder);
        var req = ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(shardLabels.addAll(LabelsFormat.parse("name=certificate.days_until_expiration, certificate=mon-cert"))))
                .build();

        var response = handler.resolveOne(req).join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.NOT_FOUND, response.getStatus());
    }

    @Test
    public void resolveMetricWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit");
        addMetrics(shard, usage, limit);
        assertEquals(usage, resolve(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage"));
        assertEquals(limit, resolve(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.limit"));
    }

    @Test
    public void resolveWhenNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);

        assertEquals(certificate, resolve(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
    }

    @Test
    public void resolveByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");
        assertEquals(certificateName, resolve(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=my"));
        assertEquals(certificateName, resolve(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob"));
    }

    @Test
    public void resolveMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        assertEquals(diskLatencyName, resolve(SHARD_COMPUTE, "name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'"));
        assertEquals(diskLatencyName, resolve(SHARD_COMPUTE, "name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'"));
    }

    @Test
    public void resolveIgnoreReplacedMetrics() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificateReplaced = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3ae6fi6fc00en0qkm9");
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificateReplaced);
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3ae6fi6fc00en0qkm9", "my", true);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my", false);
        assertEquals(certificateName, resolve(SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=my"));
    }

    private Labels resolve(int numId, String labels) {
        var shardLabels = shardResolver.resolveShard(numId).getShardKey().toLabels();
        var handler = new ResolveOneHandler(shardResolver, referenceResolver, resourceFinder);
        var req = ResolveOneRequest.newBuilder()
                .addAllLabels(LabelConverter.labelsToProtoList(shardLabels.addAll(LabelsFormat.parse(labels))))
                .build();

        var response = handler.resolveOne(req).join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());
        return ShardKeyAndMetricKey.of(LabelConverter.protoToLabels(response.getMetric().getLabelsList())).getMetricKey();
    }
}
