package ru.yandex.solomon.coremon.meta.service.handler;

import java.util.Arrays;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.protobuf.LabelConverter;
import ru.yandex.solomon.labels.protobuf.LabelSelectorConverter;
import ru.yandex.solomon.labels.query.SelectorsFormat;
import ru.yandex.solomon.metabase.api.protobuf.EMetabaseStatusCode;
import ru.yandex.solomon.metabase.api.protobuf.TUniqueLabelsRequest;

import static org.junit.Assert.assertEquals;

/**
 * @author Vladimir Gordiychuk
 */
public class UniqueLabelsHandlerTest extends HandlerTestBase {

    @Test
    public void uniqueLabelsEmpty() {
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate=*", "certificate"));
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate=abc", "certificate"));
        assertEquals(Set.of(), uniqueLabels(SHARD_COMPUTE, "", "host"));
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "host=*", "host"));
    }

    @Test
    public void uniqueLabelsWithoutResourceReference() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var usage = LabelsFormat.parse("name=quota.certificates_count.usage, host=cluster");
        var limit = LabelsFormat.parse("name=quota.certificates_count.limit, host=cluster");
        addMetrics(shard, usage, limit);
        check(usage, SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage");
        check(usage, SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage");
        check(usage, SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.us*");
        check(limit, SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.l*");
        assertEquals(Set.of(usage, limit), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "", "name", "host"));
    }

    @Test
    public void uniqueLabelsWhenResourceNameUnknown() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        addMetrics(shard, certificate);

        check(certificate, SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        check(certificate, SHARD_CERTIFICATE_MANAGER, "certificate=fd3cdv65706qam9l4cob");
        check(certificate, SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=*");
        check(certificate, SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration");
        check(certificate, SHARD_CERTIFICATE_MANAGER, "");
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate!=fd3cdv65706qam9l4cob", "name", "certificate"));
    }

    @Test
    public void uniqueLabelsByNameBeforeMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my");

        check(certificateName, SHARD_CERTIFICATE_MANAGER, "certificate=my");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "certificate=*");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
    }

    @Test
    public void uniqueLabelsByNameAfterMetabase() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=mon-cert");
        addMetrics(shard, certificate);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "mon-cert");

        check(certificateName, SHARD_CERTIFICATE_MANAGER, "name=certificate.days_until_expiration");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "name=certificate*");
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "name=quota.certificates_count.usage", "name", "certificate"));
    }

    @Test
    public void uniqueLabelsMultipleReference() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatency = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");

        addMetrics(shard, diskLatency);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");

        check(diskLatencyName, SHARD_COMPUTE, "");
        check(diskLatencyName, SHARD_COMPUTE, "name=disk_write_latency, resource_id=solomon-stockpile-sas-01");
        check(diskLatencyName, SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=hdd-01");
        check(diskLatencyName, SHARD_COMPUTE, "name=disk_write_latency, device=hdd-01");
        check(diskLatencyName, SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device=epdplhd148habqj0np92");
        check(diskLatencyName, SHARD_COMPUTE, "name=disk_write_latency, device=epdplhd148habqj0np92");
    }

    @Test // TODO: decide what todo with negative match by name or id, maybe need limit match by resource_id to only exact match (gordiychuk@)
    public void uniqueLabelsByNotName() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyOneName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");
        var diskLatencyTwoName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-16', bin='100', device='hdd-16'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");

        check(diskLatencyTwoName, SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-01");
        check(diskLatencyTwoName, SHARD_COMPUTE, "name=disk_write_latency, resource_id=*, device!=hdd-01");
        check(diskLatencyOneName, SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16");
        check(diskLatencyOneName, SHARD_COMPUTE, "name=disk_write_latency, device!=hdd-16");
        check(diskLatencyOneName, SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16, device!=hdd-16");
        check(diskLatencyOneName, SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-16");
        check(diskLatencyTwoName, SHARD_COMPUTE, "name=disk_write_latency, resource_id!=solomon-stockpile-sas-0*");
    }

    @Test
    public void uniqueLabelsIgnoreReplacedMetrics() {
        var shard = shardResolver.resolveShard(SHARD_CERTIFICATE_MANAGER);
        var certificateReplaced = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3ae6fi6fc00en0qkm9");
        var certificate = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob");
        var certificateName = LabelsFormat.parse("name=certificate.days_until_expiration, certificate=my");
        addMetrics(shard, certificateReplaced);
        addResource(shard, "certificate", "fd3ae6fi6fc00en0qkm9", "my", true);
        addResource(shard, "certificate", "fd3cdv65706qam9l4cob", "my", false);

        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate=my", "certificate"));
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate=m*", "certificate"));
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "certificate=*", "certificate"));
        assertEquals(Set.of(), uniqueLabels(SHARD_CERTIFICATE_MANAGER, "", "certificate"));

        addMetrics(shard, certificate);
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "certificate=my");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "certificate=m*");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "certificate=*");
        check(certificateName, SHARD_CERTIFICATE_MANAGER, "");
    }

    @Test
    public void uniqueLabelsMultipleValue() {
        var shard = shardResolver.resolveShard(SHARD_COMPUTE);
        var diskLatencyOne = LabelsFormat.parse("name='disk_write_latency', resource_id='epdn6uas6ujoonvd99ck', bin='100', device='epdplhd148habqj0np92'");
        var diskLatencyOneName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-01', bin='100', device='hdd-01'");
        var diskLatencyTwo = LabelsFormat.parse("name='disk_write_latency', resource_id='epdhh61pg5vnviq7sfkd', bin='100', device='epd9629qg0o6offba1m7'");
        var diskLatencyTwoName = LabelsFormat.parse("name='disk_write_latency', resource_id='solomon-stockpile-sas-16', bin='100', device='hdd-16'");

        addMetrics(shard, diskLatencyOne);
        addMetrics(shard, diskLatencyTwo);
        addResource(shard, "vm", "epdn6uas6ujoonvd99ck", "solomon-stockpile-sas-01");
        addResource(shard, "vm", "epdhh61pg5vnviq7sfkd", "solomon-stockpile-sas-16");
        addResource(shard, "disk", "epdplhd148habqj0np92", "hdd-01");
        addResource(shard, "disk", "epd9629qg0o6offba1m7", "hdd-16");

        assertEquals(Set.of(diskLatencyOneName, diskLatencyTwoName), uniqueLabels(SHARD_COMPUTE, "name=disk_write_latency, device=hdd-01|hdd-16", labelNames(diskLatencyOneName)));
        assertEquals(Set.of(diskLatencyOneName, diskLatencyTwoName), uniqueLabels(SHARD_COMPUTE, "name=disk_write_latency, resource_id=epdn6uas6ujoonvd99ck|epdhh61pg5vnviq7sfkd", labelNames(diskLatencyOneName)));
    }

    private void check(Labels expectLabels, int numId, String selectors) {
        while (!expectLabels.isEmpty()) {
            expectLabels = expectLabels.removeByIndex(ThreadLocalRandom.current().nextInt(expectLabels.size()));
            var names = labelNames(expectLabels);

            var expect = Set.of(expectLabels);
            assertEquals(expect, uniqueLabels(numId, selectors, names));
        }
    }

    private Set<Labels> uniqueLabels(int numId, String selectors, String... labels) {
        var handler = new UniqueLabelsHandler(shardResolver, referenceResolver, resourceFinder);
        var response = handler.uniqueLabels(TUniqueLabelsRequest.newBuilder()
                .setShardId(numId)
                .addAllSelectors(LabelSelectorConverter.selectorsToProto(SelectorsFormat.parse(selectors)))
                .addAllNames(Arrays.asList(labels))
                .build())
                .join();
        assertEquals(response.getStatusMessage(), EMetabaseStatusCode.OK, response.getStatus());

        var uniqueList = response.getLabelListsList()
                .stream()
                .map(LabelConverter::protoToLabels)
                .collect(Collectors.toList());
        var result = Set.copyOf(uniqueList);
        assertEquals(uniqueList.toString(), uniqueList.size(), result.size());
        return result;
    }

    private String[] labelNames(Labels labels) {
        return labels.stream()
                .map(Label::getKey)
                .toArray(String[]::new);
    }
}
