package ru.yandex.solomon.coremon.meta.ttl;


import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.solomon.core.conf.ShardKeyAndId;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.CoremonMetricHelper;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.TestMetricsCollection;
import ru.yandex.solomon.coremon.meta.ttl.Batcher.DeleteBatch;
import ru.yandex.solomon.coremon.meta.ttl.Batcher.LoadMetaBatch;
import ru.yandex.solomon.coremon.meta.ttl.Batcher.LoadResourceBatch;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.util.time.InstantUtils;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static org.junit.Assert.assertArrayEquals;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;


/**
 * @author Sergey Polovko
 */
public class BatcherTest {

    private static final ShardKeyAndId KEY = new ShardKeyAndId(
            new ShardKey("solomon", "test", "experiment"),
            "solomon_test_experiment_shard_id",
            42
    );

    private static final Comparator<CoremonMetric> METRIC_COMPARATOR = (s1, s2) -> {
        if (s1.getShardId() != s2.getShardId()) {
            return Integer.compare(s1.getShardId(), s2.getShardId());
        }
        return StockpileLocalId.compare(s1.getLocalId(), s2.getLocalId());
    };

    @Test
    public void emptyCollection() {
        TestMetricsCollection<CoremonMetric> metrics = new TestMetricsCollection<>();
        Batcher batcher = new Batcher(KEY, metrics, InstantUtils.currentTimeSeconds(), Set.of(), new UnknownReferenceTrackerNoop());
        Assert.assertNull(batcher.nextBatch());
    }

    @Test
    public void split() {
        CoremonMetric[] metrics = {
            newMetric(1, 11),

            newMetric(2, 21),
            newMetric(2, 22),

            newMetric(3, 31),
            newMetric(3, 32),
            newMetric(3, 33),

            newMetric(4, 41),
            newMetric(4, 42),
            newMetric(4, 43),
            newMetric(4, 44)
        };

        TestMetricsCollection<CoremonMetric> collection = new TestMetricsCollection<>(metrics);
        Batcher batcher = new Batcher(KEY, collection, 2, InstantUtils.currentTimeSeconds(), Set.of(), new UnknownReferenceTrackerNoop());

        List<Batch> batches = collectBatches(batcher);

        // after get last batch nextBatch() must always return null
        Assert.assertNull(batcher.nextBatch());
        Assert.assertNull(batcher.nextBatch());

        // check batches count per shard
        Map<Integer, List<StockpileBatch>> batchesByShardId = batches.stream()
                .map(StockpileBatch.class::cast)
            .collect(Collectors.groupingBy(StockpileBatch::getStockpileShardId));
        assertEquals(1, batchesByShardId.get(1).size());
        assertEquals(1, batchesByShardId.get(2).size());
        assertEquals(2, batchesByShardId.get(3).size());
        assertEquals(2, batchesByShardId.get(4).size());

        // check that all metrics was seen
        CoremonMetric[] seenMetrics = batches.stream()
            .flatMap(Batch::stream)
            .sorted(METRIC_COMPARATOR)
            .toArray(CoremonMetric[]::new);

        Arrays.sort(metrics, METRIC_COMPARATOR);
        assertEquals(metrics.length, seenMetrics.length);
        for (int i = 0; i < metrics.length; i++) {
            CoremonMetricHelper.assertEquals(metrics[i], seenMetrics[i]);
        }
    }

    @Test
    public void batchTypes() {
        final int expirationTime = InstantUtils.currentTimeSeconds();

        TestMetricsCollection<CoremonMetric> metrics = new TestMetricsCollection<>(
            newMetric(1, 1, expirationTime - 1), // expired
            newMetric(1, 2, expirationTime),     // expired
            newMetric(1, 3, expirationTime + 1), // fresh
            newMetric(1, 4));                    // unknown

        Batcher batcher = new Batcher(KEY, metrics, expirationTime, Set.of(), new UnknownReferenceTrackerNoop());

        List<Batch> batches = collectBatches(batcher);
        assertEquals(1, batches.size());

        Batch batch = batches.get(0);
        assertEquals(3, batch.size());
        Assert.assertTrue(batch instanceof Batcher.LoadMetaBatch);

        assertEquals(1, batch.getMetric(0).getLocalId());
        assertEquals(4, batch.getMetric(1).getLocalId());
        assertEquals(2, batch.getMetric(2).getLocalId());
    }

    @Test
    public void update() {
        final int expirationTime = InstantUtils.currentTimeSeconds();

        TestMetricsCollection<CoremonMetric> metrics = new TestMetricsCollection<>(
            newMetric(1, 11),
            newMetric(1, 12),
            newMetric(2, 21),
            newMetric(2, 22),
            newMetric(3, 31),
            newMetric(3, 32));

        Batcher batcher = new Batcher(KEY, metrics, expirationTime, Set.of(), new UnknownReferenceTrackerNoop());
        final int expiredShardId;
        final long[] expiredLocalIds;

        // (1) expired
        {
            StockpileBatch batch = (StockpileBatch) batcher.nextBatch();
            Assert.assertTrue(batch instanceof Batcher.LoadMetaBatch);

            expiredShardId = batch.getStockpileShardId();
            expiredLocalIds = getLocalIds(batch);

            for (int i = 0; i < batch.size(); i++) {
                final CoremonMetric metric = batch.getMetric(i);
                metric.setLastPointSeconds(expirationTime); // make metric expired
            }
            batcher.update((Batcher.LoadMetaBatch) batch);
        }

        // (2) fresh
        {
            Batch batch = batcher.nextBatch();
            Assert.assertTrue(batch instanceof Batcher.LoadMetaBatch);
            for (int i = 0; i < batch.size(); i++) {
                batch.getMetric(i).setLastPointSeconds(expirationTime + 1); // make metric fresh
            }
            batcher.update((Batcher.LoadMetaBatch) batch);
        }

        // (3) unknown
        {
            Batch batch = batcher.nextBatch();
            Assert.assertTrue(batch instanceof Batcher.LoadMetaBatch);
            batcher.update((Batcher.LoadMetaBatch) batch);
        }

        var batch = (StockpileBatch) batcher.nextBatch();
        Assert.assertTrue(batch instanceof Batcher.DeleteBatch);
        assertEquals(expiredShardId, batch.getStockpileShardId());
        Assert.assertArrayEquals(expiredLocalIds, getLocalIds(batch));

        Assert.assertNull(batcher.nextBatch());
        Assert.assertNull(batcher.nextBatch());

    }

    @Test
    public void removeMetricsOnReplacedResource() {
        final int expirationTime = InstantUtils.currentTimeSeconds();

        var metrics = List.of(
                newMetric(1, 1, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob", expirationTime + 10),
                newMetric(1, 2, "name=certificate.created_at, certificate=fd3cdv65706qam9l4cob", expirationTime + 10),
                newMetric(1, 3, "name=certificate.days_until_expiration, certificate=epdplhd148habqj0np92", expirationTime + 10),
                newMetric(1, 4, "name=certificate.created_at, certificate=epdplhd148habqj0np92", expirationTime + 10),
                newMetric(1, 5, "name=certificate.days_until_expiration, certificate=fd3usstiaspv8s0jaqbt", expirationTime + 10),
                newMetric(1, 6, "name=certificate.days_until_expiration, certificate=fd3ne49i89n9hrkfqkfu", expirationTime + 10),
                newMetric(1, 7, "name=certificate.days_until_expiration, certificate=fd3tpfj1od76oisf5etp", expirationTime + 10),
                newMetric(1, 8, "name=certificate.quota_usage", expirationTime + 10)
        );

        Batcher batcher = new Batcher(KEY, new TestMetricsCollection<>(metrics), expirationTime, Set.of("certificate"), new UnknownReferenceTrackerNoop());
        // (1) load resources
        {
            var batch = (LoadResourceBatch) batcher.nextBatch();
            assertNotNull(batch);
            assertEquals(Set.of("fd3cdv65706qam9l4cob", "epdplhd148habqj0np92", "fd3usstiaspv8s0jaqbt", "fd3ne49i89n9hrkfqkfu", "fd3tpfj1od76oisf5etp"), batch.resourceIds());
            assertEquals(7, batch.size());

            var actual = batch.stream()
                    .sorted(Comparator.comparingLong(CoremonMetric::getLocalId))
                    .collect(Collectors.toList());

            assertArrayEquals(metrics.subList(0, 7).toArray(), actual.toArray());

            // uknown fd3cdv65706qam9l4cob

            // epdplhd148habqj0np92 replaced and deleted
            batch.addResource(staticResource()
                    .setResourceId("epdplhd148habqj0np92")
                    .setName("alice")
                    .setDeletedAt(System.currentTimeMillis())
                    .setReplaced(true));

            // fd3usstiaspv8s0jaqbt replaced
            batch.addResource(staticResource()
                    .setResourceId("fd3usstiaspv8s0jaqbt")
                    .setName("bob")
                    .setReplaced(true));

            // fd3ne49i89n9hrkfqkfu deleted
            batch.addResource(staticResource()
                    .setResourceId("fd3ne49i89n9hrkfqkfu")
                    .setName("eva")
                    .setDeletedAt(System.currentTimeMillis()));

            // fd3tpfj1od76oisf5etp alive
            batch.addResource(staticResource()
                    .setResourceId("fd3tpfj1od76oisf5etp")
                    .setName("mark"));

            batcher.update(batch);
        }

        // (2) delete metrics
        {
            var batch = (DeleteBatch) batcher.nextBatch();
            assertNotNull(batch);
            assertEquals(1, batch.getStockpileShardId());
            assertEquals(2, batch.size());

            var actual = batch.stream().sorted(Comparator.comparingLong(CoremonMetric::getLocalId)).collect(Collectors.toList());
            assertArrayEquals(metrics.subList(2, 4).toArray(), actual.toArray());
        }

        assertNull(batcher.nextBatch());
        assertNull(batcher.nextBatch());

        assertEquals(metrics.size(), batcher.size());
        assertEquals(6, batcher.getSkipped());
        assertEquals(2, batcher.getUnknownReference());
    }

    @Test
    public void removeReplacedReferenceAfterTsCheck() {
        final int expirationTime = InstantUtils.currentTimeSeconds();

        var metrics = List.of(
                newMetric(1, 1, "name=certificate.days_until_expiration, certificate=fd3cdv65706qam9l4cob", CoremonMetric.UNKNOWN_LAST_POINT_SECONDS),
                newMetric(1, 2, "name=certificate.created_at, certificate=fd3cdv65706qam9l4cob", CoremonMetric.UNKNOWN_LAST_POINT_SECONDS),
                newMetric(1, 3, "name=certificate.created_at, certificate=epdplhd148habqj0np92", CoremonMetric.UNKNOWN_LAST_POINT_SECONDS),
                newMetric(1, 4, "name=certificate.quota_usage", CoremonMetric.UNKNOWN_LAST_POINT_SECONDS)
        );

        Batcher batcher = new Batcher(KEY, new TestMetricsCollection<>(metrics), expirationTime, Set.of("certificate"), new UnknownReferenceTrackerNoop());
        // (1) load meta
        {
            var batch = (LoadMetaBatch) batcher.nextBatch();
            assertNotNull(batch);
            assertEquals(4, batch.size());

            var actual = batch.stream()
                    .sorted(Comparator.comparingLong(CoremonMetric::getLocalId))
                    .collect(Collectors.toList());

            assertArrayEquals(metrics.toArray(), actual.toArray());

            for (int index = 0; index < batch.size(); index++) {
                batch.getMetric(index).setLastPointSeconds(expirationTime + 10);
            }

            batcher.update(batch);
        }

        // (2) load resources
        {
            var batch = (LoadResourceBatch) batcher.nextBatch();
            assertNotNull(batch);
            assertEquals(Set.of("fd3cdv65706qam9l4cob", "epdplhd148habqj0np92"), batch.resourceIds());
            assertEquals(3, batch.size());

            var actual = batch.stream()
                    .sorted(Comparator.comparingLong(CoremonMetric::getLocalId))
                    .collect(Collectors.toList());

            assertArrayEquals(metrics.subList(0, 3).toArray(), actual.toArray());

            // uknown fd3cdv65706qam9l4cob

            // epdplhd148habqj0np92 replaced and deleted
            batch.addResource(staticResource()
                    .setResourceId("epdplhd148habqj0np92")
                    .setName("alice")
                    .setDeletedAt(System.currentTimeMillis())
                    .setReplaced(true));

            batcher.update(batch);
        }

        // (3) delete metrics
        {
            var batch = (DeleteBatch) batcher.nextBatch();
            assertNotNull(batch);
            assertEquals(1, batch.getStockpileShardId());
            assertEquals(1, batch.size());

            var actual = batch.stream().sorted(Comparator.comparingLong(CoremonMetric::getLocalId)).collect(Collectors.toList());
            assertArrayEquals(metrics.subList(2, 3).toArray(), actual.toArray());
        }

        assertNull(batcher.nextBatch());
        assertNull(batcher.nextBatch());

        assertEquals(metrics.size(), batcher.size());
        assertEquals(3, batcher.getSkipped());
        assertEquals(2, batcher.getUnknownReference());
    }

    private static List<Batch> collectBatches(Batcher batcher) {
        List<Batch> batches = new ArrayList<>();
        Batch batch;
        while ((batch = batcher.nextBatch()) != null) {
            assertNotNull(batch);
            Assert.assertTrue(!batch.isEmpty());
            Assert.assertTrue(batch.size() <= batcher.getMaxBatchSize());
            batches.add(batch);
        }
        return batches;
    }

    private static long[] getLocalIds(Batch batch) {
        return batch.stream()
            .mapToLong(CoremonMetric::getLocalId)
            .sorted()
            .toArray();
    }

    private static CoremonMetric newMetric(int shardId, long localId) {
        Labels labels = Labels.of("sensor", shardId + "-" + localId);
        return new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
    }

    private static CoremonMetric newMetric(int shardId, long localId, int lastPointSeconds) {
        Labels labels = Labels.of("sensor", shardId + "-" + localId);
        FileCoremonMetric metric = new FileCoremonMetric(shardId, localId, labels, MetricType.DGAUGE);
        metric.setLastPointSeconds(lastPointSeconds);
        return metric;
    }

    private static CoremonMetric newMetric(int shardId, long localId, String labels, int lastPointSeconds) {
        FileCoremonMetric metric = new FileCoremonMetric(shardId, localId, LabelsFormat.parse(labels), MetricType.DGAUGE);
        metric.setLastPointSeconds(lastPointSeconds);
        return metric;
    }
}
