package ru.yandex.solomon.coremon.meta.ttl.tasks;

import java.time.Instant;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;

import org.junit.Before;
import org.junit.Test;

import ru.yandex.monlib.metrics.MetricType;
import ru.yandex.solomon.core.conf.ShardKeyAndId;
import ru.yandex.solomon.coremon.meta.CoremonMetric;
import ru.yandex.solomon.coremon.meta.FileCoremonMetric;
import ru.yandex.solomon.coremon.meta.MetricsCollection;
import ru.yandex.solomon.coremon.meta.TestMetricsCollection;
import ru.yandex.solomon.coremon.meta.ttl.Batcher;
import ru.yandex.solomon.coremon.meta.ttl.ResourceLoaderStub;
import ru.yandex.solomon.coremon.meta.ttl.UnknownReferenceTrackerNoop;
import ru.yandex.solomon.labels.LabelsFormat;
import ru.yandex.solomon.labels.shard.ShardKey;
import ru.yandex.solomon.name.resolver.client.Resource;
import ru.yandex.stockpile.client.shard.StockpileLocalId;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.name.resolver.client.ResourcesTestSupport.staticResource;

/**
 * @author Vladimir Gordiychuk
 */
public class RunningTaskTest {

    private static final ShardKeyAndId KEY = new ShardKeyAndId(
            new ShardKey("solomon", "test", "experiment"),
            "solomon_test_experiment_shard_id",
            42
    );

    private final ExecutorService executor = ForkJoinPool.commonPool();
    private MetaLoaderStub loader;
    private ResourceLoaderStub resourceLoader;
    private DeleterStub deleter;

    @Before
    public void setUp() throws Exception {
        loader = new MetaLoaderStub();
        deleter = new DeleterStub();
        resourceLoader = new ResourceLoaderStub();
    }

    @Test
    public void deleteAllMetricsFromSmallShard() {
        FileCoremonMetric[] metrics = {
                newMetric("a=b"),
                newMetric("c=d"),
        };

        loader.expectLoad(metrics);
        deleter.expectDelete(metrics);
        runTask(metrics, Instant.now(), Set.of()).join();

        loader.ensureAllExpectedLoaded();
        deleter.ensureAllExpectedDeleted();
    }

    @Test
    public void tsKnownAliveRefUnknown() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        int createdAtSeconds = expireSec + 10;
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=unknown", createdAtSeconds)};

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(1, stats.getUnknownReference());
    }

    @Test
    public void tsKnownExpiredRefUnknown() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=unknown", expireSec - 10)};
        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.minusSeconds(10));
        deleter.expectDelete(metrics);

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        deleter.ensureAllExpectedDeleted();
        assertEquals(1, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsKnownAliveRefAlive() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=alive", expireSec + 10)};

        resourceLoader.addResource(newResource("alive"));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsKnownAliveRefAliveReplaced() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=replaced", expireSec + 10)};

        resourceLoader.addResource(newResource("replaced").setReplaced(true));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsKnownAliveRefDeleted() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=deleted", expireSec + 10)};

        resourceLoader.addResource(newResource("deleted").setDeletedAt(System.currentTimeMillis()));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsKnownAliveRefDeletedReplaced() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=replacedDeleted", expireSec + 10)};

        resourceLoader.addResource(newResource("replacedDeleted").setDeletedAt(System.currentTimeMillis()).setReplaced(true));
        deleter.expectDelete(metrics);

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(1, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsKnownAliveRefAbsent() {
        var expireTime = Instant.now();
        var expireSec = (int) expireTime.getEpochSecond();
        FileCoremonMetric[] metrics = {newMetric("name=test", expireSec + 10)};

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownAliveRefUnknown() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=unknown")};
        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(1, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownExpiredRefUnknown() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=unknown")};
        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.minusSeconds(10));
        deleter.expectDelete(metrics);

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        deleter.ensureAllExpectedDeleted();
        assertEquals(1, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownAliveRefAlive() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=alive")};

        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));
        resourceLoader.addResource(newResource("alive"));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownAliveRefAliveReplaced() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=replaced")};

        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));
        resourceLoader.addResource(newResource("replaced").setReplaced(true));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownKnownAliveRefDeleted() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=deleted")};

        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));
        resourceLoader.addResource(newResource("deleted").setDeletedAt(System.currentTimeMillis()));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownAliveRefDeletedReplaced() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=ref, ref=replacedDeleted")};

        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));
        resourceLoader.addResource(newResource("replacedDeleted").setDeletedAt(System.currentTimeMillis()).setReplaced(true));
        deleter.expectDelete(metrics);

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        deleter.ensureAllExpectedDeleted();
        assertEquals(1, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    @Test
    public void tsUnknownAliveRefAbsent() {
        var expireTime = Instant.now();
        FileCoremonMetric[] metrics = {newMetric("name=test")};

        loader.expectLoad(metrics);
        loader.setLatestTs(metrics[0].getLabels(), expireTime.plusSeconds(10));

        var stats = runTask(metrics, expireTime, Set.of("ref")).join();
        loader.ensureAllExpectedLoaded();
        assertEquals(0, stats.getDeletedMetrics());
        assertEquals(1, stats.getTotalMetrics());
        assertEquals(0, stats.getUnknownReference());
    }

    private Resource newResource(String id) {
        return staticResource().setResourceId(id).setName("name-" + id);
    }

    private FileCoremonMetric newMetric(String labels) {
        return newMetric(labels, FileCoremonMetric.UNKNOWN_LAST_POINT_SECONDS);
    }

    private FileCoremonMetric newMetric(String labels, int createdAtSeconds) {
        var metric = new FileCoremonMetric(1, StockpileLocalId.random(), LabelsFormat.parse(labels), createdAtSeconds, MetricType.DGAUGE);
        metric.setLastPointSeconds(createdAtSeconds);
        return metric;
    }

    private MetricsCollection<CoremonMetric> newCollection(CoremonMetric... metrics) {
        return new TestMetricsCollection<>(metrics);
    }

    private CompletableFuture<TaskStats> runTask(FileCoremonMetric[] metrics, Instant expireTime, Set<String> ref) {
        var batcher = new Batcher(KEY, newCollection(metrics), (int) expireTime.getEpochSecond(), ref, new UnknownReferenceTrackerNoop());
        var task = new RunningTask("test", 42, batcher,
                loader, resourceLoader, deleter, executor, 10);
        return task.start().thenApply(ignore -> task.getStats());
    }

}
