#include "http.h"

#include <solomon/services/dataproxy/lib/api_impl/dataproxy_service.h>
#include <solomon/services/dataproxy/lib/cluster_map/cluster_map.h>
#include <solomon/services/dataproxy/lib/config/config.h>
#include <solomon/services/dataproxy/lib/datasource/chain/chain.h>
#include <solomon/services/dataproxy/lib/datasource/comparison/comparison.h>
#include <solomon/services/dataproxy/lib/datasource/lts/lts.h>
#include <solomon/services/dataproxy/lib/datasource/merge_sts_lts/merge_sts_lts.h>
#include <solomon/services/dataproxy/lib/datasource/monitor/monitor.h>
#include <solomon/services/dataproxy/lib/datasource/sts/sts.h>
#include <solomon/services/dataproxy/lib/datasource/tsdb/placement.h>
#include <solomon/services/dataproxy/lib/datasource/tsdb/placement_config.h>
#include <solomon/services/dataproxy/lib/datasource/tsdb/tsdb.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/datasource_actor.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/dynamic_aggregation.h>
#include <solomon/services/dataproxy/lib/datasource/yasm/yasm.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/lib/metrics_page/metrics_page.h>

#include <solomon/libs/cpp/actors/config/log_component.pb.h>
#include <solomon/libs/cpp/actors/metrics/actor_runtime_metrics.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/clients/tsdb/rpc.h>
#include <solomon/libs/cpp/config/units.h>
#include <solomon/libs/cpp/config_includes/config_includes.h>
#include <solomon/libs/cpp/grpc/executor/limits.h>
#include <solomon/libs/cpp/grpc/server/server.h>
#include <solomon/libs/cpp/host_resolver/host_resolver.h>
#include <solomon/libs/cpp/http/client/curl/client.h>
#include <solomon/libs/cpp/minidump/minidump.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/selfmon/service/proto_page.h>
#include <solomon/libs/cpp/signals/signals.h>
#include <solomon/libs/cpp/yasm/shard_config/shard_config.h>

#include <infra/yasm/common/groups/metagroup_groups.h>

#include <library/cpp/actors/core/process_stats.h>
#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/grpc/server/grpc_server.h>
#include <library/cpp/monlib/metrics/metric_registry.h>
#include <library/cpp/svnversion/svnversion.h>

#include <util/stream/output.h>
#include <util/string/cast.h>
#include <util/system/getpid.h>

#include <utility>

using namespace NSolomon;
using namespace NDataProxy;

using namespace NMonitoring;
using ::NGrpc::TGRpcServer;
using yandex::solomon::config::rpc::TGrpcServerConfig;
using yandex::solomon::config::http::HttpServerConfig;

std::unique_ptr<NActors::IActor> CreateApiCache(const TDataProxyConfig& config, std::shared_ptr<IMetricRegistry> metrics) {
    if (!config.HasApiCache()) {
        Cerr << "WARNING! DataProxy started with disabled API cache" << Endl;
        return CreateMessageCacheStub();
    }
    return CreateMessageCache(config.GetApiCache(), std::move(metrics), ELogComponent::GrpcApi, "api");
}

THolder<TGRpcServer> StartGrpcServer(
        const TGrpcServerConfig& config,
        TActorRuntime& actorRuntime,
        NActors::TActorId apiCache,
        const TClusterMap& clusterMap,
        const TString& serviceId,
        IDataSourcePtr dataSource,
        TMetricRegistry& registry)
{
    Y_ENSURE(config.PortSize() == 1, "Exactly one port must be specified for ApiServer");

    ui32 port = config.GetPort(0);

    ui32 executorPool = actorRuntime.FindExecutorByName(config.GetThreadPoolName());
    ui32 traceExecutorPool = actorRuntime.FindExecutorByName(config.GetTracingThreadPoolName());
    auto& actorSystem = actorRuntime.ActorSystem();
    auto service = DataProxyService(actorSystem, registry, apiCache, executorPool, traceExecutorPool, clusterMap, serviceId, std::move(dataSource));

    auto apiServer = NSolomon::NGrpc::MakeGRpcServerFromConfig(actorSystem, registry, config);
    apiServer->AddService(std::move(service));
    apiServer->Start();

    Cerr << "API server listening on grpc://[::]:" << port << Endl;
    return apiServer;
}

TIntrusivePtr<IDataSource> CreateShortTermSource(
        TActorRuntime& actorRuntime,
        const TShortTermSourceConfig& config,
        NMonitoring::TMetricRegistry& registry,
        const TClusterMap& clusterMap,
        const TString& clientId)
{
    return ShortTermSource(actorRuntime, config, clusterMap, registry, clientId);
}

NTsdb::TTsdbClientConfig CreateTsdbConfig() {
    TCurlClientOptions curlOpts;
    curlOpts.Name = "tsdb-client";
    curlOpts.WorkerThreads = 2;
    curlOpts.HandlerThreads = 2;

    // TODO(ivanzhukov): change to a better value
    curlOpts.DnsCacheLifetime = TDuration::Max();

    curlOpts.MaxInflight = 1000;
    curlOpts.QueueSizeLimit = 1000;

    TRequestOpts requestOpts;
    requestOpts.ReadTimeout = TDuration::Seconds(15);
    requestOpts.ConnectTimeout = TDuration::Seconds(5);

    return {curlOpts, requestOpts};
}

TIntrusivePtr<IDataSource> CreateTsdbSource(
        TActorRuntime& actorRuntime,
        const TTsdbSourceConfig& config,
        NMonitoring::TMetricRegistry& registry,
        const TString& clientId)
{
    const auto& groupPlacementFile = config.GetGroupPlacementFile();
    Y_VERIFY(!groupPlacementFile.empty(), "TsdbSource::GetGroupPlacementFile must be specified");
    const auto port = config.GetTsdbConfig().GetPort();
    Y_VERIFY(port, "TsdbConfig::Port must be specified");

    auto groupToHosts = ParseGroupPlacementFile(groupPlacementFile);

    yandex::solomon::config::rpc::TCircuitBreakerConfig circuitBreakerConfig;
    if (config.GetTsdbConfig().HasCircuitBreakerConfig()) {
        circuitBreakerConfig = config.GetTsdbConfig().GetCircuitBreakerConfig();
    } else {
        circuitBreakerConfig.set_failurequantilethreshold(0.4);
        auto time = circuitBreakerConfig.mutable_resettimeout();
        time->set_value(30);
        time->set_unit(yandex::solomon::config::SECONDS);
    }

    auto tsdbCluster = NTsdb::CreateTsdbClusterRpc(port, clientId, CreateTsdbConfig(), registry, std::move(circuitBreakerConfig));
    for (const auto& [group, hosts]: *groupToHosts) {
        for (const auto& host: hosts) {
            tsdbCluster->Add(host);
        }
    }

    auto fsCacheLifetime = config.HasFsCacheLifetime() ? FromProtoTime(config.GetFsCacheLifetime()) : TDuration::Zero();
    auto actor = PlacementActor(groupToHosts, tsdbCluster, config.GetFsCachePath(), fsCacheLifetime);
    auto placementActor = actorRuntime.Register(actor.release());
    return TsdbSource(actorRuntime, std::move(groupToHosts), std::move(tsdbCluster), placementActor, registry);
}

IDataSourcePtr CreateDataSource(
        TActorRuntime& actorRuntime,
        const TDataProxyConfig& config,
        const TClusterMap& clusterMap,
        std::shared_ptr<TMetricRegistry> metrics)
{
    ::NYasm::NGroups::TMetagroupGroupsConfig mgConfig;
    auto metagroupToGroups = std::make_shared<const THashMap<TString, TVector<TString>>>(mgConfig.GetMetagroupToGroupsListProdMap(true));
    TYasmConfig yasmConfig{std::move(metagroupToGroups), NSolomon::NYasm::CreateDefaultShardConfig()};

    TVector<IDataSourcePtr> dataSources; // the order of elements is important!

    auto ltsDataSource = LongTermSource(
            actorRuntime,
            config.GetLongTermSource(),
            clusterMap,
            metrics,
            config.GetClientId());

    TIntrusivePtr<IDataSource> baseDataSource;
    if (config.HasShortTermSource() || config.HasTsdbSource()) {
        auto createSolomonSts = [&]() {
            return CreateShortTermSource(
                    actorRuntime,
                    config.GetShortTermSource(),
                    *metrics,
                    clusterMap,
                    config.GetClientId());
        };
        auto createYasmSts = [&]() {
            return CreateTsdbSource(actorRuntime, config.GetTsdbSource(), *metrics, config.GetClientId());
        };

        TIntrusivePtr<IDataSource> stsDataSource;
        if (config.HasShortTermSource() && config.HasTsdbSource()) {
            stsDataSource = CreateComparisonSource(
                    actorRuntime,
                    createSolomonSts(),
                    createYasmSts(),
                    *metrics,
                    config.GetDefaultShortTermStorage(),
                    config.GetIsComparisonEnabled());
        } else if (config.HasShortTermSource()) {
            stsDataSource = createSolomonSts();
        } else {
            stsDataSource = createYasmSts();
        }

        NSelfMon::RegisterPage(
                actorRuntime.ActorSystem(),
                "/metrics",
                "Metrics",
                CreateMetricsPage(actorRuntime, yasmConfig, stsDataSource, ltsDataSource, *metrics));

        // [base] == [merge] <-- ( [sts], [lts] )
        baseDataSource =
                NMerger::MergeShortLongTermsSource(actorRuntime, std::move(stsDataSource), std::move(ltsDataSource));
    } else {
        // [base] == [lts]
        baseDataSource = std::move(ltsDataSource);
    }

    auto aggrDataSource = YasmDynAggrDataSource(baseDataSource, &actorRuntime.ActorSystem());
    auto yasmDataSourceActor =
            actorRuntime.Register(YasmDataSourceActor(std::move(yasmConfig), aggrDataSource, *metrics));

    // [yasm] <-- [dynaggr] <-- [base]
    dataSources.push_back(YasmDataSource(actorRuntime, yasmDataSourceActor, std::move(aggrDataSource)));
    dataSources.push_back(baseDataSource);

    return MonitorDataSource(DataSourceChain(dataSources), *metrics);
}

void ResolveAllAddresses(TDataProxyConfig& config) {
    auto replace = [r = CachingResolver(ConductorGroupResolver())](auto* addresses) {
        TAddressSet resolved;
        for (const TString& a: *addresses) {
            Cerr << "resolving " << a << Endl;
            r->Resolve(a, &resolved);
        }
        addresses->Clear();
        for (const TString& a: resolved) {
            addresses->Add()->assign(a);
        }
    };

    auto replaceReplica = [&](TReplicaConfig* rc) {
        if (rc->HasSas()) {
            replace(rc->MutableSas()->MutableAddresses());
        }
        if (rc->HasVla()) {
            replace(rc->MutableVla()->MutableAddresses());
        }
        if (rc->HasMan()) {
            replace(rc->MutableMan()->MutableAddresses());
        }
        if (rc->HasMyt()) {
            replace(rc->MutableMyt()->MutableAddresses());
        }
        if (rc->HasIva()) {
            replace(rc->MutableIva()->MutableAddresses());
        }
    };

    auto* clusterMap = config.MutableClusterMap();
    if (clusterMap->HasR0()) {
        replaceReplica(clusterMap->MutableR0());
    }
    if (clusterMap->HasR1()) {
        replaceReplica(clusterMap->MutableR1());
    }

    if (config.HasShortTermSource()) {
        auto* stsConfig = config.MutableShortTermSource();
        replace(stsConfig->MutableMemStoreClient()->MutableAddresses());
    }

    if (config.HasLongTermSource()) {
        auto* ltsConfig = config.MutableLongTermSource();
        replace(ltsConfig->MutableMetabaseConfig()->MutableClient()->MutableAddresses());
        replace(ltsConfig->MutableStockpileConfig()->MutableClient()->MutableAddresses());
    }
}

void FallbackValuesForCache(TCacheConfig* cache) {
    if (!cache->HasProjectLimit()) {
        auto* pl = cache->MutableProjectLimit();
        pl->SetUnit(yandex::solomon::config::MEGABYTES);
        pl->SetValue(20);
    }

    if (!cache->HasTotalLimit()) {
        auto* tl = cache->MutableTotalLimit();
        tl->SetUnit(yandex::solomon::config::GIGABYTES);
        tl->SetValue(2);
    }

    if (!cache->HasExpireAfter()) {
        auto* ea = cache->MutableExpireAfter();
        ea->SetUnit(yandex::solomon::config::MINUTES);
        ea->SetValue(10);
    }

    if (!cache->HasRefreshInterval()) {
        auto* ri = cache->MutableRefreshInterval();
        ri->SetUnit(yandex::solomon::config::MINUTES);
        ri->SetValue(10);
    }
}

void FallbackValuesForStockpileConfig(TDataProxyConfig& config) {
    auto* stockpileConfig = config.MutableLongTermSource()->MutableStockpileConfig();

    if (stockpileConfig->GetRequestInflightPerShard() == 0) {
        stockpileConfig->SetRequestInflightPerShard(10000);
    }
}

void FallbackValuesForMetabaseConfig(TDataProxyConfig& config) {
    auto* metabaseConfig = config.MutableLongTermSource()->MutableMetabaseConfig();

    if (metabaseConfig->GetRequestInflightPerShard() == 0) {
        metabaseConfig->SetRequestInflightPerShard(2000);
    }

    auto* shardTtl = metabaseConfig->MutableShardTtl();
    bool unitIsNotSpecified = shardTtl->GetUnit() == yandex::solomon::config::TimeUnit_MIN; // even if it is specified,
    // min value, i.e. nanoseconds, doesn't make sense in this context
    if (shardTtl->GetValue() == 0 || unitIsNotSpecified) {
        shardTtl->SetValue(5);
        shardTtl->SetUnit(yandex::solomon::config::MINUTES);
    }

    auto* shardsCleanupInterval = metabaseConfig->MutableShardsCleanupInterval();
    if (shardsCleanupInterval->GetValue() == 0) {
        shardsCleanupInterval->SetValue(1);
        shardsCleanupInterval->SetUnit(yandex::solomon::config::MINUTES);
    }

    FallbackValuesForCache(metabaseConfig->MutableCache());
}

int Main(const TDataProxyConfig& config) {
    NSolomon::NGrpc::SetThreadsLimit(2);

    const TClusterMap clusterMap(config.GetClusterMap());
    auto metrics = std::make_shared<TMetricRegistry>();

    // actor system
    auto actorRuntime = TActorRuntime::Create(config.GetActorSystem(), metrics);
    actorRuntime->Start();
    AddActorRuntimeMetrics(*actorRuntime, *metrics);
    metrics->IntGauge({{"sensor", "version"}, {"revision", GetArcadiaLastChange()}})->Set(1);

    // http server (also used for exporting metrics, so start as soon as possible)
    auto httpServer = config.HasMonServer()
            ? StartHttpServer(*actorRuntime, metrics, config.GetMonServer())
            : std::unique_ptr<NSolomon::NHttp::THttpServer>();

    NSelfMon::RegisterPage(
            actorRuntime->ActorSystem(),
            "/config", "Config",
            NSelfMon::ProtoPage(std::make_shared<TDataProxyConfig>(config)));

    // datasources
    auto dataSource = CreateDataSource(*actorRuntime, config, clusterMap, metrics);

    dataSource->WaitUntilInitialized();

    // API cache
    auto apiCache = actorRuntime->Register(
            CreateApiCache(config, metrics),
            NActors::TMailboxType::HTSwap,
            actorRuntime->FindExecutorByName(config.GetApiServer().GetThreadPoolName()));

    // gRPC server
    auto apiServer = StartGrpcServer(
            config.GetApiServer(),
            *actorRuntime,
            apiCache,
            clusterMap,
            config.GetClientId(),
            dataSource,
            *metrics);

    // signals handling loop
    sigset_t blockMask;
    SigEmptySet(&blockMask);

    while (true) {
        SigSuspend(&blockMask);

        if (NeedTerminate) {
            actorRuntime->EmergencyLog("DataProxy was terminated");

            // destroy in reverse order
            apiServer->Stop();
            apiServer.Reset();

            // destroy here all core actors
            auto future = actorRuntime->AsyncPoison(std::set{apiCache}, TDuration::Seconds(5));
            future.GetValueSync();

            actorRuntime->Stop();
            actorRuntime.Reset();

            httpServer.reset();

            return 0;
        } else if (NeedReopenLog) {
            NeedReopenLog = 0;
            actorRuntime->ReopenLog();
            TLogBackend::ReopenAllBackends();
        }
    }
}

int main(int argc, const char* argv[]) {
    NLastGetopt::TOpts opts;
    opts.AddLongOption("config", "path to configuration file")
        .Required()
        .RequiredArgument("PATH");
    opts.AddVersionOption();
    opts.AddHelpOption();
    opts.SetFreeArgsNum(0);

    try {
        NLastGetopt::TOptsParseResult r(&opts, argc, argv);

        InitSignals();
        Cerr << "Starting DataProxy (pid: " << GetPID() << ", SVN revision: " << GetArcadiaLastChange() << ')' << Endl;

        auto config = ParseTextProto(r.Get("config"));
        MergeIncludes(config);
        Validate(config);

        SetupMinidump(config.minidump_config());

        ResolveAllAddresses(config);
        FallbackValuesForMetabaseConfig(config);
        FallbackValuesForStockpileConfig(config);

        if (config.HasApiCache()) {
            FallbackValuesForCache(config.MutableApiCache());
        }

        return Main(config);
    } catch (const NLastGetopt::TUsageException&) {
        opts.PrintUsage("dataproxy");
    } catch (const TConfigParseException& e) {
        Cerr << "Cannot parse config: " << e.AsStrBuf() << Endl;
    } catch (const TConfigValidationException& e) {
        Cerr << "Config is not valid: " << e.AsStrBuf() << Endl;
    } catch (...) {
        Cerr << "Unhandled exception: " << CurrentExceptionMessage() << ". Process will terminate" << Endl;
    }

    return 1;
}
