import argparse
import grpc
import time
import types
import json

from solomon.services.dataproxy.api import dataproxy_service_pb2
from solomon.protos.math import operation_pb2
from solomon.services.dataproxy.api import dataproxy_service_pb2_grpc
from solomon.services.dataproxy.ctl.helper import ParseDataReadManyResponse
from google.protobuf import text_format


def proto_to_text(msg):
    return text_format.MessageToString(msg)


def now_millis():
    return int(time.time() * 1000)


# single shard request
def req_find(stub):
    req = dataproxy_service_pb2.FindRequest()
    req.project_id = "solomon"
    req.selectors = '{cluster="production", service="coremon", host="fetcher-sas-001", sensor="jvm.threadPool.activeThreads"}'
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 20
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call Find() method")
    resp = stub.Find(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


# cross shard request
def req_find2(stub):
    req = dataproxy_service_pb2.FindRequest()
    req.project_id = "solomon"
    req.selectors = '{cluster="production", service="*", host="cluster", sensor="proc.self.rssBytes"}'
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 20
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call Find() method")
    resp = stub.Find(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_find(stub):
    req = dataproxy_service_pb2.FindRequest()
    req.project_id = "yasm_pull_sts1_balancer"
    req.selectors = '{hosts="IVA.005", signal="balancer_report-report-monitorings-outgoing_404_summ"}'
    req.from_millis = now_millis() - 15 * 1000 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 20
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call Find() method")
    resp = stub.Find(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_resolve_one(stub):
    req = dataproxy_service_pb2.ResolveOneRequest()
    req.project_id = "solomon"
    req.name = ""
    req.labels.extend(['project', 'solomon', 'cluster', 'production', 'service', 'coremon', 'sensor', 'proc.self.rssBytes', 'host', 'Sas'])
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call ResolveOne() method")
    resp = stub.ResolveOne(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_label_keys(stub):
    req = dataproxy_service_pb2.LabelKeysRequest()
    req.project_id = "solomon"
    req.selectors = '{cluster="production", service="coremon"}'
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()

    print("call LabelKeys() method")
    resp = stub.LabelKeys(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_label_keys(stub):
    req = dataproxy_service_pb2.LabelKeysRequest()
    req.project_id = "yasm_pull_sts1_yasmtsdb"
    req.selectors = '{}'
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()

    print("call LabelKeys() method")
    resp = stub.LabelKeys(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_label_keys_hosts(stub):
    req = dataproxy_service_pb2.LabelKeysRequest()
    req.project_id = "yasm_pull_sts1_yasmtsdb"
    req.selectors = '{hosts="ASEARCH"}'
    req.from_millis = now_millis() - 15 * 60 * 1000  # -15s
    req.to_millis = now_millis()

    print("call LabelKeys() method")
    resp = stub.LabelKeys(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_label_values(stub):
    req = dataproxy_service_pb2.LabelValuesRequest()
    req.project_id = "solomon"
    req.selectors = '{cluster="production", service="coremon"}'
    req.keys.extend(['sensor'])
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 30

    print("call LabelValues() method")
    resp = stub.LabelValues(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_label_values_hosts(stub):
    req = dataproxy_service_pb2.LabelValuesRequest()
    req.project_id = "yasm_pull_sts1_yasmtsdb"
    req.selectors = '{hosts="IVA.005"}'
    req.keys.extend(['hosts'])
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 100

    print("call LabelValues() method")
    resp = stub.LabelValues(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_label_values(stub):
    req = dataproxy_service_pb2.LabelValuesRequest()
    req.project_id = "yasm_pull_sts1_yasmtsdb"
    req.selectors = '{hosts="ASEARCH"}'
    req.keys.extend(['signal'])
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 30

    print("call LabelValues() method")
    resp = stub.LabelValues(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_yasm_label_values_all(stub):
    req = dataproxy_service_pb2.LabelValuesRequest()
    req.project_id = "yasm_pull_sts1_yasmtsdb"
    req.selectors = '{hosts="ASEARCH"}'
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.limit = 100

    print("call LabelValues() method")
    resp = stub.LabelValues(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_read_one(stub):
    req = dataproxy_service_pb2.ReadOneRequest()
    req.project_id = "solomon"
    req.name = ""
    req.labels.extend(['project', 'solomon', 'cluster', 'production', 'service', 'coremon', 'sensor', 'proc.self.rssBytes', 'host', 'Sas'])
    req.max_time_series_format = 38
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call ReadOne() method")
    resp = stub.ReadOne(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


def req_read_many(stub):
    req = dataproxy_service_pb2.ReadManyRequest()
    req.project_id = "solomon"
    req.lookup.selectors = '{cluster="production", service="coremon", host="fetcher-sas-001", sensor="jvm.threadPool.activeThreads"}'
    req.lookup.limit = 100
    req.from_millis = now_millis() - 15 * 60 * 1000
    req.to_millis = now_millis()
    req.max_time_series_format = 38
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call ReadMany() method")
    resp = stub.ReadMany(req)
    print("response: {\n" + proto_to_text(resp) + "}\n")


# yasm read many request
def req_yasm_read_many(stub):
    req = dataproxy_service_pb2.ReadManyRequest()
    req.project_id = "yasm_antirobot"
    req.lookup.selectors = "{project='yasm_antirobot', ctype='prestable|prod', hosts='ASEARCH', signal='unistat_daemon-block_responses_total_deee', geo='vla'}"
    req.lookup.limit = 10000

    req.max_time_series_format = 39
    req.short_term_storage = dataproxy_service_pb2.TSDB
    operation = operation_pb2.Operation()
    operation.downsampling.grid_millis = 5 * 1000
    req.operations.extend([operation])
    req.from_millis = now_millis() - 25 * 60 * 1000
    req.to_millis = now_millis()
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call ReadMany() method")
    resp = stub.ReadMany(req)
    print("response: \n" + json.dumps(ParseDataReadManyResponse(resp), indent=2) + "\n")


def req_yasm_read_many_hist(stub):
    req = dataproxy_service_pb2.ReadManyRequest()
    req.project_id = "yasm_yasmsrv"
    req.lookup.selectors = "{geo='man', hosts='ASEARCH', signal='unistat-realtime_delay_dhhh'}"
    req.max_time_series_format = 39
    req.from_millis = now_millis() - 60 * 1000
    req.to_millis = now_millis()
    print("request: {\n" + proto_to_text(req) + "}\n")

    print("call ReadMany() method")
    resp = stub.ReadMany(req)
    print("response: \n" + json.dumps(ParseDataReadManyResponse(resp), indent=2) + "\n")


def main():
    funcs = [k[4:] for k, v in globals().items() if k.startswith('req_') and isinstance(v, types.FunctionType)]

    parser = argparse.ArgumentParser()
    parser.add_argument("--server", "-s", default="localhost:5770", help="dataproxy host")
    parser.add_argument("request", choices=funcs, help="request to send")
    args = parser.parse_args()

    host = args.server
    if ':' not in host:
        host = host + ':5770'
    channel = grpc.insecure_channel(host)
    stub = dataproxy_service_pb2_grpc.DataProxyServiceStub(channel)

    cmd = args.request
    func = globals().get('req_' + cmd, None)

    func(stub)
