package main

import (
	"context"
	"sync"
	"sync/atomic"
	"time"
)

type ClientsPool struct {
	sync.Mutex
	clients []HostClient
	idx     int32
}

func NewClientsPool() *ClientsPool {
	return &ClientsPool{}
}

func (c *ClientsPool) Init(host, port string, size int) error {
	c.Lock()
	defer c.Unlock()

	if len(c.clients) >= size {
		return nil
	}

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Minute)
	defer cancel()

	for i := len(c.clients); i < size; i++ {
		client, err := NewHostClient(ctx, host, port)
		if err != nil {
			return err
		}
		c.clients = append(c.clients, client)
	}
	return nil
}

func (c *ClientsPool) GetClient() HostClient {
	size := len(c.clients)
	var nextIdx = int(atomic.AddInt32(&c.idx, 1)) % size
	if nextIdx < 0 {
		nextIdx += size
	}
	return c.clients[nextIdx]
}
