package main

import (
	"context"
	"go.uber.org/zap"
	"time"

	"github.com/yandex/pandora/cli"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/provider"
	"github.com/yandex/pandora/core/register"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	dataproxy "a.yandex-team.ru/solomon/services/dataproxy/api"
)

var clientsPool = NewClientsPool()

type GunConfig struct {
	Host    string `validate:"required"`
	Port    string `validate:"required"`
	Clients int    `validate:"required"`
}

type Gun struct {
	conf GunConfig
	aggr core.Aggregator
	deps core.GunDeps
}

func NewGun(conf GunConfig) *Gun {
	return &Gun{conf: conf}
}

func (g *Gun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	err := clientsPool.Init(g.conf.Host, g.conf.Port, g.conf.Clients)
	if err != nil {
		return err
	}

	g.aggr = aggr
	g.deps = deps
	return nil
}

func (g *Gun) Shoot(genericAmmo core.Ammo) {
	ammo := genericAmmo.(*ProtoAmmo)
	sample := netsample.Acquire(ammo.Tag())

	var (
		err  error
		code = 500
	)

	defer func() {
		sample.SetProtoCode(code)
		if err != nil {
			g.deps.Log.Info("Request failed, status", zap.Int("code", code), zap.Error(err))
			sample.SetErr(err)
		}
		g.aggr.Report(sample)
	}()

	code, err = g.write(ammo)
}

func convertResponseToStatus(resp *dataproxy.ReadManyResponse) (int, error) {
	if len(resp.Metrics) == 0 {
		return 404, nil
	}
	return 200, nil
}

func convertGrpcStatus(err error) (int, error) {
	s := status.Convert(err)
	if s == nil {
		return 500, err
	}

	switch s.Code() {
	case codes.OK:
		return 200, nil
	case codes.Canceled:
		return 499, nil
	case codes.InvalidArgument:
		return 400, err
	case codes.DeadlineExceeded:
		return 504, err
	case codes.NotFound:
		return 404, err
	case codes.AlreadyExists:
		return 409, err
	case codes.PermissionDenied:
		return 403, err
	case codes.ResourceExhausted:
		return 429, err
	case codes.FailedPrecondition:
		return 400, err
	case codes.Aborted:
		return 409, err
	case codes.OutOfRange:
		return 400, err
	case codes.Unimplemented:
		return 501, err
	case codes.Unavailable:
		return 503, err
	case codes.Unauthenticated:
		return 401, err
	default:
		return 500, err
	}
}

func (g *Gun) write(ammo *ProtoAmmo) (int, error) {
	ctx, cancel := context.WithTimeout(context.Background(), 60*time.Second)
	defer cancel()

	var client = clientsPool.GetClient()

	req := ammo.Message.(*dataproxy.ReadManyRequest)
	resp, err := client.ReadMany(ctx, req)

	if err != nil {
		return convertGrpcStatus(err)
	}

	return convertResponseToStatus(resp)
}

func main() {
	fs := coreimport.GetFs()
	coreimport.Import(fs)

	register.Provider("read_many_provider", NewAmmoProvider, provider.DefaultDecodeProviderConfig)
	register.Gun("dataproxy_gun", NewGun, func() GunConfig {
		return GunConfig{}
	})

	cli.Run()
}
