#pragma once

#include "api_context.h"
#include "api_server_events.h"
#include "grpc_errors.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/api_methods/api_method.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/hash/dataproxy_request_hash.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/grpc/stats/req_ctx_wrapper.h>
#include <solomon/libs/cpp/threading/timer/timer_wheel.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/libs/cpp/yasm/constants/client_id.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>

namespace NSolomon::NDataProxy {

template <typename TDerive>
class TBaseApiHandler: public NActors::TActorBootstrapped<TBaseApiHandler<TDerive>> {
    class TPendingRequest;
    using TPendingId = ui32;

    class TRequestTimeoutEvent: public NTimer::ITimerEvent {
    public:
        TRequestTimeoutEvent(
                const TString& reqId,
                TPendingId pendingId,
                TInstant receivedAt,
                TBaseApiHandler* handler)
            : ReqId_(reqId)
            , PendingId_(pendingId)
            , ReceivedAt_(receivedAt)
            , Handler_(handler)
        {
        }

        void Execute() override {
            std::unique_ptr<TRequestTimeoutEvent> deferDelete(this);
            auto* op = Handler_->FindOperationByReqId(ReqId_);
            if (op == nullptr) {
                return;
            }
            if (auto req = op->Extract(PendingId_); req) {
                // respond to client with an error
                TStringBuilder sb;
                sb << "request timeout after " << (NActors::TActivationContext::Now() - ReceivedAt_);
                TRACING_SPAN_END(req->Span());
                req->ReqCtx()->ReplyError(grpc::DEADLINE_EXCEEDED, sb);
            }
        }

    private:
        TString ReqId_; // For lookup in Inflight_
        TPendingId PendingId_; // For lookup in AsyncOperation list
        TInstant ReceivedAt_;
        TBaseApiHandler* Handler_;
    };

    class TPendingRequest {
    public:
        TPendingRequest(
                ::NGrpc::IRequestContextBase* reqCtx,
                const TString& reqId,
                TPendingId id,                
                TInstant receivedAt,
                NTracing::TSpanId span,
                NTimer::TTimerWheel* timer,
                TBaseApiHandler* handler)
            : ReqCtx_{std::move(reqCtx)}
            , TimeoutEvent_{nullptr}
            , Span_{std::move(span)}
        {
            auto maxDeadline = NActors::TActivationContext::Now() + MAX_API_TIMEOUT;
            TimeoutEvent_ = new TRequestTimeoutEvent{reqId, id, receivedAt, handler};
            timer->ScheduleAt(TimeoutEvent_, Min(ReqCtx_->Deadline(), maxDeadline));
        }

        ::NGrpc::IRequestContextBase* ReqCtx() {
            return ReqCtx_;
        }

        const NTracing::TSpanId& Span() {
            return Span_;
        }

        void CancelTimeout() {
            // Destructor also calls TimeoutEvent_->Cancel()
            delete TimeoutEvent_;
        }

    private:
        ::NGrpc::IRequestContextBase* ReqCtx_;
        // This event may outlive the TPendingRequest and even TAsyncOperation. It will self-destroy when fired
        TRequestTimeoutEvent* TimeoutEvent_;
        NTracing::TSpanId Span_;
    };

    struct TAsyncOperation {
        NActors::TActorId LoaderId;

        void AddPending(
                ::NGrpc::IRequestContextBase* reqCtx,
                const TString& reqId,
                TInstant receivedAt,
                NTracing::TSpanId span,
                NTimer::TTimerWheel* timer,
                TBaseApiHandler* handler)
        {
            PendingRequests_.emplace(PendingRequestId_, std::make_unique<TPendingRequest>(reqCtx, reqId, PendingRequestId_, receivedAt, std::move(span), timer, handler));
            PendingRequestId_++;
        }

        std::unique_ptr<TPendingRequest> Extract(TPendingId id) {
            auto it = PendingRequests_.find(id);
            if (it == PendingRequests_.end()) {
                return nullptr;
            }
            auto ret = std::move(it->second);
            PendingRequests_.erase(it);
            return ret;
        }

        bool Empty() const {
            return PendingRequests_.empty();
        }

        TPendingRequest* Front() {
            auto it = PendingRequests_.begin();
            if (it == PendingRequests_.end()) {
                return nullptr;
            }
            return it->second.get();
        }

        void ReplyOk(const google::protobuf::Message& message) {
            using TResponse = typename TDerive::TResponse;

            for (auto&& [id, req]: PendingRequests_) {
                TResponse copy;
                copy.CopyFrom(message);
                TRACING_SPAN_END(req->Span());
                req->ReqCtx()->Reply(&copy);
                req->CancelTimeout();
            }
            PendingRequests_.clear();
        }

        void ReplyError(grpc::StatusCode code, const TString& msg) {
            for (auto&& [id, req]: PendingRequests_) {
                TRACING_SPAN_END(req->Span());
                req->ReqCtx()->ReplyError(code, msg);
                req->CancelTimeout();
            }
            PendingRequests_.clear();
        }

    private:
        TPendingId PendingRequestId_{0};
        absl::flat_hash_map<TPendingId, std::unique_ptr<TPendingRequest>> PendingRequests_;
    };

    static constexpr TDuration DEADLINE_TIMER_TICK_INTERVAL = TDuration::Seconds(1);

public:
    TBaseApiHandler(TApiServerContext* apiCtx, EApiMethod method)
        : ApiCtx_{apiCtx}
        , Method_{method}
        , Timer_(TInstant::Now())
    {
    }

    void Bootstrap() {
        this->Become(&TBaseApiHandler::StateFunc);
        auto delayMicros = DEADLINE_TIMER_TICK_INTERVAL.MicroSeconds() / 2 + RandomNumber(DEADLINE_TIMER_TICK_INTERVAL.MicroSeconds());
        this->Schedule(TDuration::MicroSeconds(delayMicros), new NActors::TEvents::TEvWakeup);
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TApiServerEvents::TRequest, OnRequest);
            hFunc(TCacheEvents::TLookupResult, OnCacheResult);
            hFunc(TApiServerEvents::TDataLoaded, OnDataLoaded);
            hFunc(TApiServerEvents::TDataLoadError, OnDataLoadError);
            sFunc(NActors::TEvents::TEvWakeup, OnWakeup);
            sFunc(NActors::TEvents::TEvPoison, OnPoison);
        }
    }

private:
    void OnRequest(TApiServerEvents::TRequest::TPtr& ev) {
        using TRequest = typename TDerive::TRequest;

        auto* req = ev->Get();
        auto* reqCtx = req->RequestCtx;
        auto traceCtx = TRACING_NEW_SPAN_START(NGrpc::MakeTraceContextFromGrpc(reqCtx), reqCtx->GetRequest()->GetTypeName());

        auto reqDeadline = reqCtx->Deadline();
        auto nowUpperBound = Timer_.Now() + Timer_.TickDuration();
        if (reqDeadline <= nowUpperBound) {
            // request already expired, reply with DEADLINE_EXCEEDED
            TStringBuilder sb;
            sb << "request already expired, deadline (" << reqDeadline << ") <= now (" << nowUpperBound << ')';
            TRACING_SPAN_END(traceCtx);
            reqCtx->ReplyError(grpc::DEADLINE_EXCEEDED, sb);
            return;
        }

        auto softDeadline = GetSoftDeadline(*reqCtx, reqDeadline);
        if (softDeadline && *softDeadline <= nowUpperBound) {
            // soft deadline already expired, reply with INVALID_ARGUMENT
            TStringBuilder sb;
            sb << "incorrect soft deadline (" << *softDeadline << ") <= now (" << nowUpperBound << ')';
            TRACING_SPAN_END(traceCtx);
            reqCtx->ReplyError(grpc::INVALID_ARGUMENT, sb);
            return;
        }

        auto* appReq = static_cast<const TRequest*>(reqCtx->GetRequest());
        TString projectId = appReq->project_id();
        if (!projectId) {
            TRACING_SPAN_END(traceCtx);
            reqCtx->ReplyError(grpc::INVALID_ARGUMENT, "no project id given");
            return;
        }

        auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
        TStringBuf clientId = clientIds.empty() ? ""sv : clientIds[0];
        TString reqHash;
        if (clientId.StartsWith(NYasm::YASM_COLLECTOR_CLIENT_ID_PREFIX)) {
            reqHash += clientId;
            reqHash += '\t';
        }
        reqHash += Hash(*appReq);
        TString reqId = projectId + '\t' + reqHash;

        TAsyncOperation& asyncOp = InFlight_[reqId];
        bool firstReq = asyncOp.Empty();
        asyncOp.AddPending(req->RequestCtx, reqId, req->ReceivedAt, NTracing::TSpanId(traceCtx), &Timer_, this);

        if (firstReq) {
            auto lookup = std::make_unique<TCacheEvents::TLookup>();
            lookup->Project = projectId;
            lookup->MessageType = ToUnderlying(Method_);
            lookup->MessageHash = reqHash;
            auto span = TRACING_NEW_SPAN_START(traceCtx, "lookup in cache");
            this->Send(ApiCtx_->Cache, lookup.release(), 0, 0, std::move(span));
        }
    }

    void OnCacheResult(TCacheEvents::TLookupResult::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto* result = ev->Get();
        TString reqId = result->Project + '\t' + result->MessageHash;

        auto it = InFlight_.find(reqId);
        if (it == InFlight_.end()) {
            MON_ERROR(GrpcApi, "cache result for unknown request");
            return;
        }

        TAsyncOperation& asyncOp = it->second;
        if (result->Data && !result->NeedsRefresh) {
            asyncOp.ReplyOk(*result->Data);
            InFlight_.erase(it);
            return;
        }
        TPendingRequest* req = asyncOp.Front();
        if (!req) {
            // all pending requests were expired
            InFlight_.erase(it);
            return;
        }

        try {
            asyncOp.LoaderId = LoadData(std::move(reqId), req->ReqCtx(), NTracing::TSpanId(req->Span()));
        } catch (...) {
            this->Send(this->SelfId(),
                new TApiServerEvents::TDataLoadError{
                        std::move(reqId),
                        EDataSourceStatus::BAD_REQUEST,
                        "cannot parse request: " + CurrentExceptionMessage()},
                0,
                0,
                NTracing::TSpanId(req->Span()));
        }
    }

    void OnDataLoaded(TApiServerEvents::TDataLoaded::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto* result = ev->Get();

        auto it = InFlight_.find(result->ReqId);
        if (it == InFlight_.end()) {
            MON_ERROR(GrpcApi, "reply to unknown request: " << result->Data->ShortDebugString());
            return;
        }

        auto splitIdx = result->ReqId.find('\t');
        Y_VERIFY(splitIdx != TString::npos);

        TAsyncOperation& asyncOp = it->second;
        asyncOp.ReplyOk(*result->Data);
        InFlight_.erase(it);

        auto store = std::make_unique<TCacheEvents::TStore>();
        store->Project = result->ReqId.substr(0, splitIdx);
        store->MessageType = ToUnderlying(Method_);
        store->MessageHash = result->ReqId.substr(splitIdx + 1);
        store->Data = std::move(result->Data);
        this->Send(ApiCtx_->Cache, store.release());
    }

    void OnDataLoadError(TApiServerEvents::TDataLoadError::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto* result = ev->Get();

        auto it = InFlight_.find(result->ReqId);
        if (it == InFlight_.end()) {
            MON_ERROR(GrpcApi, "reply to unknown request: " << result->Status << ": " << result->ErrorMessage);
            return;
        }

        TAsyncOperation& asyncOp = it->second;
        asyncOp.ReplyError(ToGrpcStatus(result->Status), result->ErrorMessage);
        InFlight_.erase(it);

        // TODO: also cache NOT_FOUND result
    }

    void OnPoison() {
        for (auto& it: InFlight_) {
            this->Send(it.second.LoaderId, new NActors::TEvents::TEvPoison);
        }
        InFlight_.clear();
        // TODO: all not fired TimeoutEvents from Timer_ are leaked here
        // This is probably ok, since actor dies when dataproxy terminates
        this->PassAway();
    }

    void OnWakeup() {
        auto tick = NActors::TActivationContext::Now() - Timer_.Now();
        if (tick >= Timer_.TickDuration()) {
            Timer_.Advance(tick);
        }
        this->Schedule(DEADLINE_TIMER_TICK_INTERVAL, new NActors::TEvents::TEvWakeup);
    }

    NActors::TActorId LoadData(TString reqId, ::NGrpc::IRequestContextBase* reqCtx, NTracing::TSpanId traceCtx) {
        // may throw
        auto query = TDerive::MakeQuery(reqCtx);

        auto loader = static_cast<TDerive*>(this)->CreateDataLoader(ApiCtx_, this->SelfId(), std::move(reqId));
        auto loaderId = this->Register(loader.release(), NActors::TMailboxType::Simple);
        auto span = TRACING_NEW_SPAN_START(std::move(traceCtx), "Load data for " << reqCtx->GetRequest()->GetTypeName());
        this->Send(loaderId, new TDataSourceEvents::TQuery{std::move(query)}, 0, 0, std::move(span));
        return loaderId;
    }

    TAsyncOperation* FindOperationByReqId(const TString& reqId) {
        auto it = InFlight_.find(reqId);
        if (it == InFlight_.end()) {
            return nullptr;
        }
        return &it->second;
    }

private:
    TApiServerContext* ApiCtx_;
    EApiMethod Method_;
    absl::flat_hash_map<TString, TAsyncOperation, THash<TString>> InFlight_;
    NTimer::TTimerWheel Timer_;
};

} // namespace NSolomon::NDataProxy
