#include "dataproxy_service.h"
#include "find_handler.h"
#include "label_keys_handler.h"
#include "label_values_handler.h"
#include "resolve_one_handler.h"
#include "resolve_many_handler.h"
#include "metric_names_handler.h"
#include "unique_labels_handler.h"
#include "read_one_handler.h"
#include "read_many_handler.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/grpc/stats/req_ctx_wrapper.h>
#include <solomon/libs/cpp/grpc/stats/stats_filter.h>
#include <solomon/libs/cpp/grpc/stats/stats_page.h>

#include <library/cpp/grpc/server/grpc_server.h>
#include <library/cpp/grpc/server/grpc_request.h>

#include <library/cpp/actors/core/actorsystem.h>

using namespace NActors;
using namespace NGrpc;
using namespace NMonitoring;

namespace NSolomon::NDataProxy {
namespace {

using yandex::monitoring::dataproxy::DataProxyService;

/**
 * Implementation of DataProxy gRPC service
 */
class TServiceImpl final: public TGrpcServiceBase<DataProxyService> {
public:
    TServiceImpl(
            TActorSystem& actorSystem,
            TMetricRegistry& registry,
            TActorId apiCache,
            ui32 executorPool,
            ui32 traceExecutorPool,
            const TClusterMap& clusterMap,
            const TString& serviceId,
            IDataSourcePtr dataSource)
        : ActorSystem_(actorSystem)
        , ApiCtx_{clusterMap, registry, apiCache, std::move(dataSource)}
        , ExecutorPool_(executorPool)
        , StatsFilter_{CreateStatsFilter(NGrpc::TStatsFilterConf{100, {}, TDuration::Seconds(1)},
                       NGrpc::TStatsTableConf{registry, NGrpc::TStatsTableSettings{TDuration::Days(1), 50}})}
    {
        NTracing::InitTracingService(ApiCtx_.Registry, ActorSystem_, traceExecutorPool, serviceId);
        NSelfMon::RegisterPage(ActorSystem_, "/tracing", "Tracing", NTracing::TracingServiceId());
        NSelfMon::RegisterPage(ActorSystem_, "/grpc_server", "gRPC Server", NSolomon::NGrpc::StatsPage(StatsFilter_->GetStatsTable(), StatsFilter_));
    }

    void InitService(grpc::ServerCompletionQueue* cq, TLoggerPtr logger) override {
        using AS = DataProxyService::AsyncService;

        RegisterHandler2<TFindHandler>(cq, logger, &AS::RequestFind, "Find");
        RegisterHandler2<TResolveOneHandler>(cq, logger, &AS::RequestResolveOne, "ResolveOne");
        RegisterHandler2<TResolveManyHandler>(cq, logger, &AS::RequestResolveMany, "ResolveMany");
        RegisterHandler2<TLabelKeysHandler>(cq, logger, &AS::RequestLabelKeys, "LabelKeys");
        RegisterHandler2<TLabelValuesHandler>(cq, logger, &AS::RequestLabelValues, "LabelValues");
        RegisterHandler2<TMetricNamesHandler>(cq, logger, &AS::RequestMetricNames, "MetricNames");
        RegisterHandler2<TUniqueLabelsHandler>(cq, logger, &AS::RequestUniqueLabels, "UniqueLabels");
        RegisterHandler2<TReadOneHandler>(cq, logger, &AS::RequestReadOne, "ReadOne");
        RegisterHandler2<TReadManyHandler>(cq, logger, &AS::RequestReadMany, "ReadMany");
    }

    template <typename THandler, typename TCallback>
    void RegisterHandler2(grpc::ServerCompletionQueue* cq, TLoggerPtr logger, TCallback callback, const char* name) {
        auto handlerId = ActorSystem_.Register(new THandler(&ApiCtx_), TMailboxType::HTSwap, ExecutorPool_);

        using TReq = typename THandler::TRequest;
        using TResp = typename THandler::TResponse;

        auto request = MakeIntrusive<TGRpcRequest<TReq, TResp, TServiceImpl>>(
            this,
            TServiceImpl::GetService(),
            cq,
            [this, handlerId, StatsFilterPtr = StatsFilter_](auto* reqCtx) {
                ActorSystem_.Send(handlerId, new TApiServerEvents::TRequest{StatsFilterPtr->FilterRequest(reqCtx, ActorSystem_.Timestamp()), ActorSystem_.Timestamp()});
            },
            callback,
            name,
            logger,
            FakeCounterBlock());
        request->Run();
    }

    void SetGlobalLimiterHandle(TGlobalLimiter* limiter) final {
        Limiter_ = limiter;
    }

    bool IncRequest() {
        return Limiter_->Inc();
    }

    void DecRequest() {
        Limiter_->Dec();
    }

private:
    TActorSystem& ActorSystem_;
    TApiServerContext ApiCtx_;
    const ui32 ExecutorPool_;
    TGlobalLimiter* Limiter_ = nullptr;
    std::shared_ptr<NGrpc::IStatsFilter> StatsFilter_;
};

} // namespace

TIntrusivePtr<IGRpcService> DataProxyService(
        TActorSystem& actorSystem,
        TMetricRegistry& registry,
        NActors::TActorId apiCache,
        ui32 executorPool,
        ui32 traceExecutorPool,
        const TClusterMap& clusterMap,
        const TString& serviceId,
        IDataSourcePtr dataSource)
{
    return MakeIntrusive<TServiceImpl>(actorSystem, registry, apiCache, executorPool, traceExecutorPool, clusterMap, serviceId, std::move(dataSource));
}

} // namespace NSolomon::NDataProxy
