#include "find_handler.h"
#include "api_server_events.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/cluster_map/cluster_map.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/api/dataproxy_service.grpc.pb.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::FindRequest;
using yandex::monitoring::dataproxy::FindResponse;
using ::NGrpc::IRequestContextBase;

class TFindLoader: public TActor<TFindLoader> {
public:
    TFindLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TFindLoader>{&TFindLoader::StateFunc}
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TFindQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TFindResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

private:
    void OnRequest(TDataSourceEvents::TQuery<TFindQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->Find");
        ApiCtx_->DataSource->Find(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TFindResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TFindResult>::TPtr& ev) {
        try {
            auto resp = std::make_unique<FindResponse>();

            auto result = std::move(ev->Get()->Result);
            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* metrics = resp->mutable_metrics();
            metrics->Reserve(static_cast<int>(result->Metrics.size()));

            for (const TMetric<ui32>& m: result->Metrics) {
                auto* mProto = metrics->Add();
                mProto->set_type(NSolomon::ToProto(m.Type));
                mProto->set_name_idx(m.Name);

                auto* labels = mProto->mutable_labels_idx();
                labels->Reserve(static_cast<int>(2 * m.Labels.size()));
                for (const TLabel<ui32>& l: m.Labels) {
                    auto* lProto = labels->AddNAlreadyReserved(2);
                    *lProto++ = l.Key;
                    *lProto = l.Value;
                }

                {
                    auto* mStockpileId = mProto->mutable_stockpile_replica_1();
                    mStockpileId->set_shard_id(m.StockpileIds[EReplica::R0].ShardId);
                    mStockpileId->set_local_id(m.StockpileIds[EReplica::R0].LocalId);
                }
                {
                    auto* mStockpileId = mProto->mutable_stockpile_replica_2();
                    mStockpileId->set_shard_id(m.StockpileIds[EReplica::R1].ShardId);
                    mStockpileId->set_local_id(m.StockpileIds[EReplica::R1].LocalId);
                }
            }
            resp->set_replica_name_1(TString{DcToStr(result->Dcs[EReplica::R0])}); // TODO: remove it completely from gateway
            resp->set_replica_name_2(TString{DcToStr(result->Dcs[EReplica::R1])}); // TODO: remove it completely from gateway

            resp->set_total_count(result->TotalCount);
            resp->set_truncated(result->Truncated);

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TFindQuery TFindHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const FindRequest*>(reqCtx->GetRequest());

    TFindQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Time.From = TInstant::MilliSeconds(req->from_millis());
    query.Time.To = TInstant::MilliSeconds(req->to_millis());
    query.Selectors = ParseSelectors(req->selectors());
    query.Limit = (req->limit() != 0) ? Min<ui32>(req->limit(), MAX_METRICS) : MAX_METRICS;
    query.FillMetricName = req->fill_metric_name();
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

    return query;
}

std::unique_ptr<IActor> TFindHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TFindLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
