#include "label_keys_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
//#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::LabelKeysRequest;
using yandex::monitoring::dataproxy::LabelKeysResponse;

class TLabelKeysLoader: public TActor<TLabelKeysLoader> {
public:
    TLabelKeysLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TLabelKeysLoader>(&TLabelKeysLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TLabelKeysQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TLabelKeysResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TLabelKeysQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->LabelKeys");
        ApiCtx_->DataSource->LabelKeys(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TLabelKeysResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TLabelKeysResult>::TPtr& ev) {
        try {
            auto& result = ev->Get()->Result;

            auto resp = std::make_unique<LabelKeysResponse>();
            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* keysProto = resp->mutable_keys_idx();
            keysProto->Reserve(result->Keys.size());
            for (ui32 id: result->Keys) {
                keysProto->Add(id);
            }

            for (const auto& error: result->Errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        MON_WARN(GrpcApi, "LabelKeys " << status << " (" << message << ')');

        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TLabelKeysQuery TLabelKeysHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const LabelKeysRequest*>(reqCtx->GetRequest());

    TLabelKeysQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Selectors = ParseSelectors(req->selectors());
    query.Time.From = TInstant::MilliSeconds(req->from_millis());
    query.Time.To = TInstant::MilliSeconds(req->to_millis());
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

    // TODO: restore producer
//     if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//         yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//     }

    return query;
}

std::unique_ptr<IActor> TLabelKeysHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TLabelKeysLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
