#include "label_values_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/trace/trace.h>
//#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::LabelValuesRequest;
using yandex::monitoring::dataproxy::LabelValuesResponse;

class TLabelValuesLoader: public TActor<TLabelValuesLoader> {
public:
    TLabelValuesLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TLabelValuesLoader>(&TLabelValuesLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TLabelValuesQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TLabelValuesResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TLabelValuesQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->LabelValues");
        ApiCtx_->DataSource->LabelValues(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TLabelValuesResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TLabelValuesResult>::TPtr& ev) {
        try {
            auto& result = ev->Get()->Result;

            auto resp = std::make_unique<LabelValuesResponse>();
            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* labelsProto = resp->mutable_labels();
            labelsProto->Reserve(result->Labels.size());
            for (auto& label: result->Labels) {
                auto* labelProto = labelsProto->Add();
                labelProto->set_key_idx(label.Key);

                auto* valuesProto = labelProto->mutable_values_idx();
                valuesProto->Reserve(label.Values.size());
                for (ui32 id: label.Values) {
                    valuesProto->Add(id);
                }

                labelProto->set_metric_count(label.MetricCount);
                labelProto->set_truncated(label.Truncated);
            }

            for (const auto& error: result->Errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            resp->set_metric_count(result->MetricCount);

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TLabelValuesQuery TLabelValuesHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const LabelValuesRequest*>(reqCtx->GetRequest());

    TLabelValuesQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Selectors = ParseSelectors(req->selectors());
    query.Keys.reserve(req->keys_size());
    for (const auto& key: req->keys()) {
        query.Keys.push_back(key);
    }
    query.TextFilter = req->text_filter();
    query.Time.From = TInstant::MilliSeconds(req->from_millis());
    query.Time.To = TInstant::MilliSeconds(req->to_millis());
    query.Limit = (req->limit() != 0) ? Min<ui32>(req->limit(), MAX_LABEL_VALUES) : MAX_LABEL_VALUES;
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

//  if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//      yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//  }

    return query;
}

std::unique_ptr<IActor> TLabelValuesHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TLabelValuesLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
