#include "metric_names_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/trace/trace.h>
//#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::MetricNamesRequest;
using yandex::monitoring::dataproxy::MetricNamesResponse;

class TMetricNamesLoader: public TActor<TMetricNamesLoader> {
public:
    TMetricNamesLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TMetricNamesLoader>(&TMetricNamesLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TMetricNamesQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TMetricNamesResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TMetricNamesQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->MetricNames");
        ApiCtx_->DataSource->MetricNames(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TMetricNamesResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TMetricNamesResult>::TPtr& ev) {
        try {
            auto& result = ev->Get()->Result;

            auto resp = std::make_unique<MetricNamesResponse>();
            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* namesProto = resp->mutable_names_idx();
            namesProto->Reserve(result->Names.size());
            for (ui32 id: result->Names) {
                namesProto->Add(id);
            }

            resp->set_total_count(result->TotalCount);
            resp->set_truncated(result->Truncated);

            for (const auto& error: result->Errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TMetricNamesQuery TMetricNamesHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const MetricNamesRequest*>(reqCtx->GetRequest());

    TMetricNamesQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Selectors = ParseSelectors(req->selectors());
    query.TextFilter = req->text_filter();
    query.Limit = (req->limit() != 0) ? Min<ui32>(req->limit(), MAX_METRIC_NAMES) : MAX_METRIC_NAMES;
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];
//  if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//      yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//  }

    return query;
}

std::unique_ptr<IActor> TMetricNamesHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TMetricNamesLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
