#include "read_many_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/stockpile_codec/format.h>
#include <solomon/libs/cpp/logging/logging.h>
//#include <solomon/protos/common/request_producer.pb.h>
#include <solomon/libs/cpp/grpc/stats/req_ctx_wrapper.h>
#include <solomon/libs/cpp/string_map/string_map.h>
#include <solomon/libs/cpp/trace/trace.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::ReadManyRequest;
using yandex::monitoring::dataproxy::ReadManyResponse;

// TODO: reuse this function
template <typename T>
void CopyLabels(const T& labelsProto, TLabels<ui32>* labels) {
    Y_ENSURE(labelsProto.size() % 2 == 0, "labels array must contain even number of items");
    labels->reserve(labelsProto.size() / 2);

    for (int i = 0, size = labelsProto.size(); i < size; ) {
        ui32 key = labelsProto.Get(i++);
        ui32 value = labelsProto.Get(i++);
        labels->emplace_back(key, value);
    }
}

class TReadManyLoader: public TActor<TReadManyLoader> {
public:
    TReadManyLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TReadManyLoader>(&TReadManyLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TReadManyQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TReadManyResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TReadManyQuery>::TPtr& ev) {
        auto& query = ev->Get()->Query;
        SpanId_ = std::move(ev->TraceId);
        QueryTimeSeriesFormat_ = NStockpile::FormatFromInt(query.MaxTimeSeriesFormat);

        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->ReadMany");
        ApiCtx_->DataSource->ReadMany(
                std::move(query),
                MakeReplyToHandler<TReadManyResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TReadManyResult>::TPtr& ev) {
        try {
            auto& strings = ev->Get()->Result->Strings;
            auto& metrics = ev->Get()->Result->Metrics;
            auto& errors = ev->Get()->Result->Errors;

            auto resp = std::make_unique<ReadManyResponse>();
            auto stringPool = strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* metricsProto = resp->mutable_metrics();
            metricsProto->Reserve(metrics.size());

            for (const auto& metric: metrics) {
                auto* metricData = metricsProto->Add();

                auto* metricProto = metricData->mutable_metric();
                metricProto->set_type(NSolomon::ToProto(metric.Meta.Type));
                metricProto->set_name_idx(metric.Meta.Name);

                for (auto& label: metric.Meta.Labels) {
                    metricProto->add_labels_idx(label.Key);
                    metricProto->add_labels_idx(label.Value);
                }

                if (metric.TimeSeries) {
                    ToProto(*metric.TimeSeries, metricData->mutable_time_series(), QueryTimeSeriesFormat_);
                }

                if (metric.Aggregate) {
                    ToProto(*metric.Aggregate, metricData->mutable_aggregate());
                }
            }

            for (const auto& error: errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    NStockpile::EFormat QueryTimeSeriesFormat_;
    TSpanId SpanId_;
};

} // namespace

TReadManyQuery TReadManyHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const ReadManyRequest*>(reqCtx->GetRequest());

    TReadManyQuery query;
    // TODO: move common query inflation logic to a separate function and reuse it in every *_handler:
    // FillQuery(reqCtx, query);
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Time.From = TInstant::MilliSeconds(req->from_millis());
    query.Time.To = TInstant::MilliSeconds(req->to_millis());

    switch (req->query_case()) {
        case ReadManyRequest::QueryCase::kLookup: {
            query.Lookup = std::make_unique<TReadManyQuery::TLookup>();
            query.Lookup->Selectors = ParseSelectors(req->lookup().selectors());
            query.Lookup->Limit = req->lookup().limit();
            query.Lookup->GroupBy = {req->lookup().group_by().begin(), req->lookup().group_by().end()};
            break;
        }

        case ReadManyRequest::QueryCase::kResolvedKeys: {
            query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();
            query.ResolvedKeys->Strings = NStringPool::TStringPool(req->resolved_keys().string_pool());
            query.ResolvedKeys->MetricKeys.reserve(req->resolved_keys().metric_keys().size());

            // TODO: validate common labels have cluster & service
            CopyLabels(req->resolved_keys().common_labels_idx(), &query.ResolvedKeys->CommonLabels);

            for (const auto& metricProto: req->resolved_keys().metric_keys()) {
                query.ResolvedKeys->MetricKeys.emplace_back();
                auto& metricKey = query.ResolvedKeys->MetricKeys.back();
                metricKey.Name = metricProto.name_idx();
                CopyLabels(metricProto.labels_idx(), &metricKey.Labels);
            }
            break;
        }

        case ReadManyRequest::QueryCase::QUERY_NOT_SET: {
            throw yexception() << "Query selectors or resolved keys should be specified";
        }
    }
    query.Operations.assign(req->operations().begin(), req->operations().end());
    query.MaxTimeSeriesFormat = req->max_time_series_format();
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];
//      if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//          yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//      }

    return query;
}

std::unique_ptr<IActor> TReadManyHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TReadManyLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
