#include "read_one_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/stockpile_codec/format.h>
#include <solomon/libs/cpp/trace/trace.h>
#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::ReadOneRequest;
using yandex::monitoring::dataproxy::ReadOneResponse;

class TReadOneLoader: public TActor<TReadOneLoader> {
public:
    TReadOneLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TReadOneLoader>(&TReadOneLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TReadOneQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TReadOneResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TReadOneQuery>::TPtr& ev) {
        auto& query = ev->Get()->Query;
        SpanId_ = std::move(ev->TraceId);
        QueryTimeSeriesFormat_ = NStockpile::FormatFromInt(query.MaxTimeseriesFormat);

        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->ReadOne");
        ApiCtx_->DataSource->ReadOne(
                std::move(query),
                MakeReplyToHandler<TReadOneResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TReadOneResult>::TPtr& ev) {
        try {
            auto resp = std::make_unique<ReadOneResponse>();
            auto* metricProto = resp->mutable_metric();
            auto& metric = ev->Get()->Result->Metric;
            auto errors = ev->Get()->Result->Errors;

            metricProto->set_type(NSolomon::ToProto(metric.Type));
            metricProto->set_name(std::move(metric.Name));

            for (auto& label: metric.Labels) {
                metricProto->add_labels(std::move(label.Key));
                metricProto->add_labels(std::move(label.Value));
            }

            auto& timeSeries = *ev->Get()->Result->TimeSeries;
            ToProto(timeSeries, resp->mutable_time_series(), QueryTimeSeriesFormat_);

            for (const auto& error: errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
    NStockpile::EFormat QueryTimeSeriesFormat_;
};

} // namespace

TReadOneQuery TReadOneHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const ReadOneRequest*>(reqCtx->GetRequest());

    TReadOneQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Time.From = TInstant::MilliSeconds(req->from_millis());
    query.Time.To = TInstant::MilliSeconds(req->to_millis());
    query.Name = req->name();
    query.Operations.assign(req->operations().begin(), req->operations().end());

    Y_ENSURE(req->labels_size() % 2 == 0, "labels array must contain even number of items");
    query.Labels.reserve(req->labels_size() / 2);
    for (int i = 0, size = req->labels_size(); i < size; ) {
        const auto& key = req->labels(i++);
        const auto& value = req->labels(i++);
        query.Labels.push_back(TLabel<TString>{key, value});
    }

    query.MaxTimeseriesFormat = req->max_time_series_format();
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];
//  if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//      yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//  }

    return query;
}

std::unique_ptr<IActor> TReadOneHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TReadOneLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
