#include "resolve_many_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/trace/trace.h>
//#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::ResolveManyRequest;
using yandex::monitoring::dataproxy::ResolveManyResponse;

template <typename T>
void CopyLabels(const T& labelsProto, TLabels<ui32>* labels) {
    Y_ENSURE(labelsProto.size() % 2 == 0, "labels array must contain even number of items");
    labels->reserve(labelsProto.size() / 2);

    for (int i = 0, size = labelsProto.size(); i < size; ) {
        ui32 key = labelsProto.Get(i++);
        ui32 value = labelsProto.Get(i++);
        labels->emplace_back(TLabel<ui32>{key, value});
    }
}

class TResolveManyLoader: public TActor<TResolveManyLoader> {
public:
    TResolveManyLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TResolveManyLoader>(&TResolveManyLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TResolveManyQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TResolveManyResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TResolveManyQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->ResolveMany");
        ApiCtx_->DataSource->ResolveMany(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TResolveManyResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TResolveManyResult>::TPtr& ev) {
        try {
            auto& result = ev->Get()->Result;

            auto resp = std::make_unique<ResolveManyResponse>();

            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* metricsProto = resp->mutable_metrics();
            metricsProto->Reserve(result->Metrics.size());

            for (const auto& m: result->Metrics) {
                auto* mProto = metricsProto->Add();
                mProto->set_type(NSolomon::ToProto(m.Type));
                mProto->set_name_idx(m.Name);

                auto* labelsProto = mProto->mutable_labels_idx();
                labelsProto->Reserve(m.Labels.size() * 2);

                for (const TLabel<ui32>& label: m.Labels) {
                    labelsProto->Add(label.Key);
                    labelsProto->Add(label.Value);
                }

                {
                    auto* mStockpileId = mProto->mutable_stockpile_replica_1();
                    mStockpileId->set_shard_id(m.StockpileIds[EReplica::R0].ShardId);
                    mStockpileId->set_local_id(m.StockpileIds[EReplica::R0].LocalId);
                }
                {
                    auto* mStockpileId = mProto->mutable_stockpile_replica_2();
                    mStockpileId->set_shard_id(m.StockpileIds[EReplica::R1].ShardId);
                    mStockpileId->set_local_id(m.StockpileIds[EReplica::R1].LocalId);
                }
            }

            resp->set_replica_name_1(TString{DcToStr(result->Dcs[EReplica::R0])}); // TODO: remove it completely from gateway
            resp->set_replica_name_2(TString{DcToStr(result->Dcs[EReplica::R1])}); // TODO: remove it completely from gateway

            for (const auto& error: result->Errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TResolveManyQuery TResolveManyHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const ResolveManyRequest*>(reqCtx->GetRequest());

    TResolveManyQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Strings = NStringPool::TStringPool(req->string_pool());

    // TODO: validate common labels have cluster & service
    CopyLabels(req->common_labels_idx(), &query.CommonLabels);

    query.MetricKeys.reserve(req->metric_keys_size());
    for (const auto& mProto: req->metric_keys()) {
        query.MetricKeys.emplace_back();
        auto& m = query.MetricKeys.back();
        m.Name = mProto.name_idx();
        CopyLabels(mProto.labels_idx(), &m.Labels);
    }
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

    // TODO: restore producer
//     if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//         yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//     }

    return query;
}

std::unique_ptr<IActor> TResolveManyHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TResolveManyLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
