#include "resolve_one_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/trace/trace.h>
//#include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::ResolveOneRequest;
using yandex::monitoring::dataproxy::ResolveOneResponse;

class TResolveOneLoader: public TActor<TResolveOneLoader> {
public:
    TResolveOneLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TResolveOneLoader>(&TResolveOneLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TResolveOneQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TResolveOneResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TResolveOneQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->ResolveOne");
        ApiCtx_->DataSource->ResolveOne(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TResolveOneResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TResolveOneResult>::TPtr& ev) {
        try {
            auto resp = std::make_unique<ResolveOneResponse>();

            auto* metricProto = resp->mutable_metric();
            auto metric = std::move(ev->Get()->Result->Metric);
            auto& errors = ev->Get()->Result->Errors;

            metricProto->set_name(std::move(metric.Name));
            metricProto->set_type(NSolomon::ToProto(metric.Type));

            for (auto& label: metric.Labels) {
                metricProto->add_labels(std::move(label.Key));
                metricProto->add_labels(std::move(label.Value));
            }

            {
                auto* mStockpileId = metricProto->mutable_stockpile_replica_1();
                mStockpileId->set_shard_id(metric.StockpileIds[EReplica::R0].ShardId);
                mStockpileId->set_local_id(metric.StockpileIds[EReplica::R0].LocalId);
            }
            {
                auto* mStockpileId = metricProto->mutable_stockpile_replica_2();
                mStockpileId->set_shard_id(metric.StockpileIds[EReplica::R1].ShardId);
                mStockpileId->set_local_id(metric.StockpileIds[EReplica::R1].LocalId);
            }
            auto& result = ev->Get()->Result;
            resp->set_replica_name_1(TString{DcToStr(result->Dcs[EReplica::R0])}); // TODO: remove it completely from gateway
            resp->set_replica_name_2(TString{DcToStr(result->Dcs[EReplica::R1])}); // TODO: remove it completely from gateway

            for (const auto& error: errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TResolveOneQuery TResolveOneHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const ResolveOneRequest*>(reqCtx->GetRequest());

    TResolveOneQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Key.Name = req->name();

    Y_ENSURE(req->labels_size() % 2 == 0, "labels array must contain even number of items");
    query.Key.Labels.reserve(req->labels_size() / 2);
    for (int i = 0, size = req->labels_size(); i < size; ) {
        const auto& key = req->labels(i++);
        const auto& value = req->labels(i++);
        query.Key.Labels.push_back(TLabel<TString>{key, value});
    }
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

    // TODO: restore producer
//  if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//      yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//  }

    return query;
}

std::unique_ptr<IActor> TResolveOneHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TResolveOneLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
