#include "unique_labels_handler.h"

#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/datasource/reply_to_handler.h>

#include <solomon/libs/cpp/grpc/interceptor/headers.h>
#include <solomon/libs/cpp/trace/trace.h>
// #include <solomon/protos/common/request_producer.pb.h>

namespace NSolomon::NDataProxy {
namespace {

using namespace NActors;
using namespace NTracing;
using yandex::monitoring::dataproxy::UniqueLabelsRequest;
using yandex::monitoring::dataproxy::UniqueLabelsResponse;

class TUniqueLabelsLoader: public TActor<TUniqueLabelsLoader> {
public:
    TUniqueLabelsLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId)
        : TActor<TUniqueLabelsLoader>(&TUniqueLabelsLoader::StateFunc)
        , ApiCtx_{apiCtx}
        , ParentId_{parentId}
        , ReqId_{std::move(reqId)}
    {
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TDataSourceEvents::TQuery<TUniqueLabelsQuery>, OnRequest);
            hFunc(TDataSourceEvents::TSuccess<TUniqueLabelsResult>, OnSuccess);
            hFunc(TDataSourceEvents::TError, OnError);
        }
    }

    void OnRequest(TDataSourceEvents::TQuery<TUniqueLabelsQuery>::TPtr& ev) {
        SpanId_ = std::move(ev->TraceId);
        auto span = TRACING_NEW_SPAN_START(SpanId_, "DataSource->UniqueLabels");
        ApiCtx_->DataSource->UniqueLabels(
                std::move(ev->Get()->Query),
                MakeReplyToHandler<TUniqueLabelsResult>(SelfId()),
                std::move(span));
    }

    void OnSuccess(TDataSourceEvents::TSuccess<TUniqueLabelsResult>::TPtr& ev) {
        try {
            auto& result = ev->Get()->Result;

            auto resp = std::make_unique<UniqueLabelsResponse>();
            auto stringPool = result->Strings.Build(yandex::solomon::common::StringPool_Compression_LZ4);
            resp->mutable_string_pool()->Swap(&stringPool);

            auto* labelsListProto = resp->mutable_labels();
            labelsListProto->Reserve(result->Labels.size());
            for (const TLabels<ui32>& labels: result->Labels) {
                auto* labelsProto = labelsListProto->Add();

                auto* labelsIdxProto = labelsProto->mutable_labels_idx();
                labelsIdxProto->Reserve(labels.size() * 2);
                for (const TLabel<ui32>& label: labels) {
                    labelsIdxProto->AddAlreadyReserved(label.Key);
                    labelsIdxProto->AddAlreadyReserved(label.Value);
                }
            }

            for (const auto& error: result->Errors) {
                auto* e = resp->add_errors();
                e->set_status((google::rpc::Code) ToGrpcStatus(error.Status));
                *(e->mutable_message()) = std::move(error.Message);
            }

            Send(ParentId_, new TApiServerEvents::TDataLoaded{std::move(ReqId_), std::move(resp)},
                    0, 0, TSpanId(SpanId_));
            PassAway();
        } catch (...) {
            SendErrorAndDie(EDataSourceStatus::UNKNOWN, "internal error: " + CurrentExceptionMessage());
        }
    }

    void OnError(TDataSourceEvents::TError::TPtr& ev) {
        auto* error = ev->Get();
        SendErrorAndDie(error->Status, error->Message);
    }

    void SendErrorAndDie(EDataSourceStatus status, TString message) {
        Send(ParentId_, new TApiServerEvents::TDataLoadError{std::move(ReqId_), status, std::move(message)},
                0, 0, std::move(SpanId_));
        PassAway();
    }

private:
    TApiServerContext* ApiCtx_;
    TActorId ParentId_;
    TString ReqId_;
    TSpanId SpanId_;
};

} // namespace

TUniqueLabelsQuery  TUniqueLabelsHandler::MakeQuery(::NGrpc::IRequestContextBase* reqCtx) {
    auto* req = static_cast<const UniqueLabelsRequest*>(reqCtx->GetRequest());

    TUniqueLabelsQuery query;
    query.Deadline = Min(reqCtx->Deadline(), TActivationContext::Now() + MAX_API_TIMEOUT);
    query.SoftDeadline = GetSoftDeadline(*reqCtx, query.Deadline);
    query.ForceReplicaRead = req->force_replica_read();
    query.Project = req->project_id();
    query.Selectors = ParseSelectors(req->selectors());

    query.Keys.reserve(req->keys_size());
    for (const auto& key: req->keys()) {
        query.Keys.push_back(key);
    }
    query.ShortTermStorage = req->short_term_storage();
    auto clientIds = reqCtx->GetPeerMetaValues(CLIENT_ID_HEADER_NAME);
    query.ClientId = clientIds.empty() ? ""sv : clientIds[0];

//  if (auto producerStr = GetFirstMetaValue(PRODUCER_HEADER_NAME)) {
//      yandex::solomon::common::RequestProducer_Parse(TString{*producerStr}, &query.Producer);
//  }

    return query;
}

std::unique_ptr<IActor> TUniqueLabelsHandler::CreateDataLoader(TApiServerContext* apiCtx, TActorId parentId, TString reqId) const {
    return std::make_unique<TUniqueLabelsLoader>(apiCtx, parentId, std::move(reqId));
}

} // namespace NSolomon::NDataProxy
