#include <solomon/services/dataproxy/lib/api_impl/grpc_meta_service.h>
#include <solomon/services/dataproxy/lib/limits.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/generic/set.h>
#include <util/string/cast.h>

#include <unordered_map>

using namespace NSolomon::NDataProxy;

class TRequestContextWrapper: public NGrpc::IRequestContextBase {
public:
    TInstant Deadline() const override {
        return TInstant::Now() + MAX_API_TIMEOUT;
    }

    TVector<TStringBuf> GetPeerMetaValues(TStringBuf key) const override {
        const auto range = MetaMap_.equal_range(key.data());
        if (range.first == range.second) {
            return {};
        }

        TVector<TStringBuf> values;
        values.reserve(std::distance(range.first, range.second));

        for (auto it = range.first; it != range.second; ++it) {
            values.emplace_back(it->second.data(), it->second.size());
        }
        return values;
    }

    void SetPeerMetaValue(TStringBuf key, TStringBuf value) {
        MetaMap_.emplace(key, value);
    }

    void Clear() {
        MetaMap_.clear();
    }

    // Override pure virtual methods
    void Reply(NProtoBuf::Message*, ui32) override {
    }

    void Reply(grpc::ByteBuffer*, ui32) override {
    }

    void ReplyUnauthenticated(const TString&) override {
    }

    void ReplyError(grpc::StatusCode, const TString&, const TString& = "") override {
    }

    void AddTrailingMetadata(const TString&, const TString&) override {
    }

    void UseDatabase(const TString&) override {
    }

    void SetNextReplyCallback(TOnNextReply&&) override {
    }

    void FinishStreamingOk() override {
    }

    const NProtoBuf::Message* GetRequest() const override {
        return nullptr;
    }

    NGrpc::TAuthState& GetAuthState() override {
        return AuthState_;
    }

    TSet<TStringBuf> GetPeerMetaKeys() const override {
        return TSet<TStringBuf>();
    }

    grpc_compression_level GetCompressionLevel() const override {
        return GRPC_COMPRESS_LEVEL_NONE;
    }

    google::protobuf::Arena* GetArena() override {
        return nullptr;
    }

    TAsyncFinishResult GetFinishFuture() override {
        return FinishPromise_.GetFuture();
    }

    TString GetPeer() const override {
        return "";
    }

    bool SslServer() const override {
        return false;
    }

private:
    std::unordered_multimap<TString, TString> MetaMap_;
    NGrpc::TAuthState AuthState_ = NGrpc::TAuthState(false);
    NThreading::TPromise<EFinishStatus> FinishPromise_;
};

TEST(TSoftDeadlineTest, EmptyFieldTest) {
    TRequestContextWrapper requestWrapper;
    TInstant deadLine = TInstant::Minutes(5);
    EXPECT_FALSE(GetSoftDeadline(requestWrapper, deadLine).has_value()) << "found non-existent value";
}

TEST(TSoftDeadlineTest, NegativeTest) {
    TRequestContextWrapper requestWrapper;
    requestWrapper.SetPeerMetaValue("x-mon-soft-deadline", "99999");
    TInstant smallDeadline = TInstant::MilliSeconds(0);
    EXPECT_FALSE(GetSoftDeadline(requestWrapper, smallDeadline).has_value()) << "soft deadline is bigger than hard";
}

TEST(TSoftDeadlineTest, PositiveTest) {
    TRequestContextWrapper requestWrapper;
    TInstant correctSoft = TInstant::Minutes(1);
    TInstant correctHard = TInstant::Minutes(5);
    requestWrapper.SetPeerMetaValue("x-mon-soft-deadline", ToString(correctSoft.MilliSeconds()));
    EXPECT_EQ(*GetSoftDeadline(requestWrapper, correctHard), correctSoft) << "incorrect value";
}
