#include <solomon/services/dataproxy/lib/cluster_id/cluster_id.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/cast.h>

#include <unordered_set>

using namespace NSolomon;
using namespace NDataProxy;

TEST(TClusterIdTest, Default) {
    TClusterId id;
    ASSERT_EQ(id.Dc(), EDc::Unknown);
    ASSERT_EQ(id.Replica(), EReplica::Unknown);
}

TEST(TClusterIdTest, Hasheable) {
    std::unordered_set<TClusterId> ids;
    std::vector<size_t> stdHashes;
    std::vector<size_t> arcHashes;

    for (EDc dc: KnownDcs) {
        for (EReplica replica: KnownReplicas) {
            TClusterId id{dc, replica};
            ids.insert(id);
            stdHashes.push_back(std::hash<TClusterId>{}(id));
            arcHashes.push_back(THash<TClusterId>{}(id));
        }
    }

    size_t expectedSize = KnownDcs.size() * KnownReplicas.size();

    ASSERT_EQ(ids.size(), expectedSize);
    ASSERT_EQ(stdHashes.size(), expectedSize);
    ASSERT_EQ(arcHashes.size(), expectedSize);
    ASSERT_EQ(stdHashes, arcHashes);
}

TEST(TClusterIdTest, ToString) {
    ASSERT_EQ(ToString(TClusterId{EDc::Sas, EReplica::R0}), "sas-r0");
    ASSERT_EQ(ToString(TClusterId{EDc::Vla, EReplica::R1}), "vla-r1");
    ASSERT_EQ(ToString(TClusterId{}), "unknown-unknown");
}

TEST(TClusterIdTest, FromString) {
    ASSERT_EQ(FromString<TClusterId>("man-r0"), TClusterId(EDc::Man, EReplica::R0));
    ASSERT_EQ(FromString<TClusterId>("myt-r1"), TClusterId(EDc::Myt, EReplica::R1));
    ASSERT_THROW(FromString<TClusterId>("iva-r3"), yexception);
    ASSERT_THROW(FromString<TClusterId>("klg-r0"), yexception);
    ASSERT_THROW(FromString<TClusterId>("jamel"), yexception);
}

TEST(TClusterIdTest, ToFromString) {
    for (EDc dc: KnownDcs) {
        for (EReplica replica: KnownReplicas) {
            TClusterId id{dc, replica};
            ASSERT_EQ(id, FromString<TClusterId>(ToString(id))) << "dc=" << dc << ", replica=" << replica;
        }
    }
}
