#include "config.h"

#include <solomon/libs/cpp/config/units.h>

#include <library/cpp/protobuf/util/pb_io.h>

#include <util/generic/hash_set.h>

#define DP_VALIDATE(condition, message) Y_ENSURE_EX(condition, TConfigValidationException() << message)

namespace NSolomon::NDataProxy {

TDataProxyConfig ParseTextProto(const TString& filePath) {
    TDataProxyConfig config;
    static EParseFromTextFormatOptions options; // no flags -- all bits set to 0
    try {
        ParseFromTextFormat(filePath, config, options);
    } catch (...) {
        // using options above, we should catch an error if a config has unknown fields
        throw TConfigParseException() << CurrentExceptionMessage() << Endl;
    }
    return config;
}

void Validate(const TDataProxyConfig& config) {
    THashSet<TStringBuf> ThreadPoolNames;

    // ActorSystem
    {
        DP_VALIDATE(config.HasActorSystem(), "ActorSystem{} section cannot be empty");
        const auto& asConfig = config.GetActorSystem();

        // ActorSystem/Executors
        for (const auto& e: asConfig.GetExecutors()) {
            bool inserted = ThreadPoolNames.insert(e.GetName()).second;
            DP_VALIDATE(inserted, "Two or more executors have the same name: " << e.GetName());
        }

        // ActorSystem/LogConfig
        // TODO: implement log config validation
    }

    // ApiServer
    {
        DP_VALIDATE(config.HasApiServer(), "ApiServer{} section cannot be empty");
        const auto& asConfig = config.GetApiServer();

        // ApiServer/Port
        ui32 port = asConfig.GetPort(0);
        DP_VALIDATE(port > 1024, "Port " << port << " in ApiServer{} section is invalid");

        // ApiServer/ThreadPoolName
        const auto& threadPoolName = asConfig.GetThreadPoolName();
        DP_VALIDATE(!threadPoolName.Empty(), "ThreadPoolName in ApiServer{} section is not configured");
        DP_VALIDATE(ThreadPoolNames.contains(threadPoolName),
                    "In ApiServer{} section used unknown thread pool: " << threadPoolName);

        // ApiServer/MaxMessageSize
        DP_VALIDATE(asConfig.HasMaxMessageSize(), "MaxMessageSize in ApiServer{} section is not configured");
        ui64 maxMsgSize = FromProtoDataSize(asConfig.GetMaxMessageSize());
        DP_VALIDATE(maxMsgSize > 0, "MaxMessageSize in ApiServer{} section is not configured");
    }

    // MonServer
    if (config.HasMonServer()) {
        const auto& msConfig = config.GetMonServer();

        // Monserver/Bind
        DP_VALIDATE(!msConfig.GetBind().Empty(), "Bind in MonServer{} section is not configured");
        // TODO: validate IP/hostname

        // Monserver/Port
        auto port = msConfig.GetPort();
        DP_VALIDATE(port > 1024, "Port " << port << " in MonServer{} section is invalid");

        // Monserver/ThreadPoolName
        const auto& threadPoolName = msConfig.GetThreadPoolName();
        DP_VALIDATE(!threadPoolName.Empty(), "ThreadPoolName in MonServer{} section is not configured");
        DP_VALIDATE(ThreadPoolNames.contains(threadPoolName),
                    "In MonServer{} section used unknown thread pool: " << threadPoolName);
    }
}

} // namespace NSolomon::NDataProxy
