#include "chain.h"

#include <solomon/libs/cpp/trace/trace.h>

using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

/**
 * DataSource implementation that is simply a chain of other data sources.
 * For each query selects the first data source that can handle it.
 */
class TDataSourceChain final: public IDataSource {
public:
    explicit TDataSourceChain(TVector<IDataSourcePtr> dataSources)
        : DataSources_(std::move(dataSources))
    {
    }

    /**
     * Cheсks that some of its data sources can process the given query.
     * @param query  query which will be processed
     * @return true if data source can process query, false - otherwise
     */
    bool CanHandle(const TQuery& query) const override {
        return SelectDataSource(query);
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::Find, std::move(handler), std::move(traceCtx));
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::ResolveOne, std::move(handler), std::move(traceCtx));
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::ResolveMany, std::move(handler), std::move(traceCtx));
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::MetricNames, std::move(handler), std::move(traceCtx));
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::LabelKeys, std::move(handler), std::move(traceCtx));
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::LabelValues, std::move(handler), std::move(traceCtx));
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::UniqueLabels, std::move(handler), std::move(traceCtx));
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::ReadOne, std::move(handler), std::move(traceCtx));
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        SelectAndCall(std::move(query), &IDataSource::ReadMany, std::move(handler), std::move(traceCtx));
    }

    void WaitUntilInitialized() const override {
        for (auto&& dataSource: DataSources_) {
            dataSource->WaitUntilInitialized();
        }
    }

private:
    const IDataSource* SelectDataSource(const TQuery& query) const {
        for (const auto& dataSourcePtr: DataSources_) {
            if (dataSourcePtr->CanHandle(query)) {
                return dataSourcePtr.Get();
            }
        }
        return nullptr;
    }

    template <typename TQuery, typename TMethod, typename THandlerPtr>
    void SelectAndCall(TQuery query, TMethod method, THandlerPtr handler, TSpanId traceCtx) const {
        const auto* dataSource = SelectDataSource(query);
        if (dataSource) {
            (dataSource->*method)(std::move(query), std::move(handler), std::move(traceCtx));
        } else {
            TRACING_SPAN_END(traceCtx);
            handler->OnError(std::move(query.Project), EDataSourceStatus::BACKEND_ERROR, "No DataSource found to handle the request");
        }
    }

    TVector<IDataSourcePtr> DataSources_;
};
} // namespace

TIntrusivePtr<IDataSource> DataSourceChain(TVector<IDataSourcePtr> dataSources) {
    return MakeIntrusive<TDataSourceChain>(std::move(dataSources));
}

} // namespace NDataProxy
