#include "comparator.h"
#include "is_equal.h"
#include "out.h"

#include <solomon/services/dataproxy/lib/timeseries/compressed.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <util/string/builder.h>

namespace NSolomon::NDataProxy {

TMetricData TMetricData::FromMetric(TReadManyResult::TMetricData&& metric, const NStringPool::TStringPool& strings) {
    TMetricData result;
    result.Type = metric.Meta.Type;
    result.TimeSeries = std::move(metric.TimeSeries);

    for (auto [key, value]: metric.Meta.Labels) {
        if (strings.At(key) == NLabels::LABEL_HOST && strings.At(value) != NYasm::AGGREGATED_MARKER) {
            result.AggrType = EAggregationType::Host;
            for (auto dc: KnownDcs) {
                if (strings.At(value).Contains(DcToStr(dc))) {
                    result.Dc = dc;
                    break;
                }
            }
            break;
        } else if (strings.At(key) == NLabels::LABEL_GROUP) {
            result.AggrType = EAggregationType::Group;
            for (auto dc: KnownDcs) {
                auto dcStr = TString{DcToStr(dc)};
                for (auto& c: dcStr) {
                    c = AsciiToUpper(c);
                }

                if (strings.At(value).Contains(dcStr)) {
                    result.Dc = dc;
                    break;
                }
            }
            break;
        }
    }
    return result;
}

std::unique_ptr<TReadManyResult> TMetricComparator::CopyResponse(TReadManyResult&& response, EStorageType type) {
    auto resultCopy = std::make_unique<TReadManyResult>();
    resultCopy->Errors = response.Errors;
    resultCopy->Metrics.reserve(response.Metrics.size());

    auto responseStrings = response.Strings.Build();

    for (auto&& metric: response.Metrics) {
        auto& metricCopy = resultCopy->Metrics.emplace_back();

        TMetricKey<ui32> metricKey;
        metricKey.Labels.reserve(metric.Meta.Labels.size());

        for (auto [key, value]: metric.Meta.Labels) {
            ui32 k = Strings_.Put(responseStrings[key]);
            ui32 v = Strings_.Put(responseStrings[value]);
            metricKey.Labels.emplace_back(k, v);

            ui32 k1 = resultCopy->Strings.Put(responseStrings[key]);
            ui32 v1 = resultCopy->Strings.Put(responseStrings[value]);
            metricCopy.Meta.Labels.emplace_back(k1, v1);
        }

        std::sort(metricKey.Labels.begin(), metricKey.Labels.end(), [](auto lhs, auto rhs) {
            return lhs.Key < rhs.Key;
        });

        auto metricName = responseStrings[metric.Meta.Name];
        if (metricName) {
            metricKey.Name = Strings_.Put(metricName);
            metricCopy.Meta.Name = resultCopy->Strings.Put(metricName);
        }

        metricCopy.Meta.Type = metric.Meta.Type;
        metricCopy.Meta.StockpileIds = metric.Meta.StockpileIds;
        if (metric.TimeSeries) {
            metricCopy.TimeSeries = std::make_unique<TCompressedTimeSeries>(*metric.TimeSeries);
        }
        if (metric.Aggregate) {
            metricCopy.Aggregate = std::make_unique<TAggregateVariant>(*metric.Aggregate);
        }

        auto& value = Metrics_[std::move(metricKey)];
        value[ToUnderlying(type)] = TMetricData::FromMetric(std::move(metric), responseStrings);
    }
    // TODO: handle errors

    return resultCopy;
}

void TMetricComparator::CaptureResponse(TReadManyResult&& response, EStorageType type) {
    auto responseStrings = response.Strings.Build();

    for (auto&& metric: response.Metrics) {
        TMetricKey<ui32> metricKey;
        metricKey.Labels.reserve(metric.Meta.Labels.size());
        for (auto [key, value]: metric.Meta.Labels) {
            ui32 k = Strings_.Put(responseStrings[key]);
            ui32 v = Strings_.Put(responseStrings[value]);
            metricKey.Labels.emplace_back(k, v);
        }

        std::sort(metricKey.Labels.begin(), metricKey.Labels.end(), [](auto lhs, auto rhs) {
            return lhs.Key < rhs.Key;
        });

        auto metricName = responseStrings[metric.Meta.Name];
        if (metricName) {
            metricKey.Name = Strings_.Put(metricName);
        }

        auto& value = Metrics_[std::move(metricKey)];
        value[ToUnderlying(type)] = TMetricData::FromMetric(std::move(metric), responseStrings);
    }
    // TODO: handle errors
}

void TMetricComparator::AddError(const TDataSourceError& error) {
    Y_UNUSED(error); // TODO: handle errors
}

void TMetricComparator::Compare() {
    auto strings = Strings_.Build();

    for (auto it = Metrics_.begin(); it != Metrics_.end();) {
        auto node = Metrics_.extract(it++);
        Compare(node.key(), std::move(node.mapped()), strings);
    }
}

TString KeyToString(const TMetricKey<ui32>& key, const NStringPool::TStringPool& strings) {
    TStringBuilder builder;
    if (key.Name == 0) {
        builder << "<empty name>";
    } else {
        builder << strings.At(key.Name);
    }

    builder << " {";
    for (auto [k, v]: key.Labels) {
        builder << strings.At(k) << '=' << strings.At(v) << ',';
    }
    builder << '}';

    return builder;
}

// looking for present and, preferably, non-empty metric to determine DC, type, etc.
std::function<void(TComparisonMetric&)> MakeRateInc(TMetricData& memStoreMetric, TMetricData& tsdbMetric) {
    Y_VERIFY(memStoreMetric.TimeSeries || tsdbMetric.TimeSeries);
    auto memStoreInc = [&memStoreMetric](TComparisonMetric& rate) {
        rate.Inc(memStoreMetric);
    };
    auto tsdbInc = [&tsdbMetric](TComparisonMetric& rate) {
        rate.Inc(tsdbMetric);
    };

    if (!tsdbMetric.TimeSeries) {
        return memStoreInc;
    }

    if (!memStoreMetric.TimeSeries) {
        return tsdbInc;
    }

    if (memStoreMetric.TimeSeries->PointCount() == 0) {
        return tsdbInc;
    } else {
        return memStoreInc;
    }
}

void TMetricComparator::Compare(
        const TMetricKey<ui32>& key,
        TMetricArray&& metrics,
        const NStringPool::TStringPool& strings)
{
    auto& memStoreMetric = metrics[ToUnderlying(EStorageType::MemStore)];
    auto& tsdbMetric = metrics[ToUnderlying(EStorageType::Tsdb)];

    // for now, TSDB datasource returns empty timeseries even if metric is missing
    if (!memStoreMetric.TimeSeries && (!tsdbMetric.TimeSeries || tsdbMetric.TimeSeries->PointCount() == 0)) {
        return;
    }

    auto inc = MakeRateInc(memStoreMetric, tsdbMetric);

    if (!memStoreMetric.TimeSeries) {
        MON_INFO(Comparison, "[MemStore] missing metric: " << KeyToString(key, strings));

        inc(MemStoreMissingMetrics_);
        inc(TsdbTotal_);
        return;
    }

    if (!tsdbMetric.TimeSeries) {
        MON_INFO(Comparison, "[TSDB] missing metric: " << KeyToString(key, strings));

        inc(TsdbMissingMetrics_);
        inc(MemStoreTotal_);
        return;
    }

    Y_VERIFY(
            tsdbMetric.Dc == memStoreMetric.Dc,
            "[TSDB] %s != [MemStore] %s; %s",
            DcToStr(tsdbMetric.Dc).data(),
            DcToStr(memStoreMetric.Dc).data(),
            KeyToString(key, strings).data());

    Y_VERIFY(tsdbMetric.AggrType == memStoreMetric.AggrType);

    inc(MemStoreTotal_);
    inc(TsdbTotal_);

    const ITimeSeries& memStoreTs = *memStoreMetric.TimeSeries;
    const ITimeSeries& tsdbTs = *tsdbMetric.TimeSeries;

    if (memStoreTs.PointCount() == 0 || tsdbTs.PointCount() == 0) {
        if (tsdbTs.PointCount() != 0) {
            MON_INFO(Comparison, "[MemStore] empty metric: " << KeyToString(key, strings));
            inc(MemStoreEmpty_);
        } else if (memStoreTs.PointCount() != 0){
            MON_INFO(Comparison, "[TSDB] empty metric: " << KeyToString(key, strings));
            inc(TsdbEmpty_);
        }
    } else if (memStoreTs.Type() != tsdbTs.Type()) {
        MON_INFO(
                Comparison,
                "type mismatch: [MemStore]" << NMonitoring::MetricTypeToStr(memStoreTs.Type()) << " != [TSDB]"
                                            << NMonitoring::MetricTypeToStr(tsdbTs.Type())
                                            << "; metric: " << KeyToString(key, strings));
        inc(TypeMismatch_);
    } else {
        Y_VERIFY(memStoreMetric.Type == tsdbMetric.Type);
        inc(PresentMetrics_);

        ComparePoints(key, std::move(memStoreMetric), std::move(tsdbMetric), strings);
    }
}

void TMetricComparator::ComparePoints(
        const TMetricKey<ui32>& key,
        TMetricData&& memStoreMetric,
        TMetricData&& tsdbMetric,
        const NStringPool::TStringPool& strings)
{
    auto inc = MakeRateInc(memStoreMetric, tsdbMetric);

    NTs::TVariantPoint memStorePoint;
    NTs::TVariantPoint tsdbPoint;

    auto memStoreIter = memStoreMetric.TimeSeries->Iterator();
    auto tsdbIter = tsdbMetric.TimeSeries->Iterator();

    bool memStoreHasNext = memStoreIter->Next(&memStorePoint);
    bool tsdbHasNext = tsdbIter->Next(&tsdbPoint);

    bool hasDiffPoints = false;
    bool isEqual = true;

    TStringBuilder pointsDiff;
    while (memStoreHasNext && tsdbHasNext) {
        if (memStorePoint.Time == tsdbPoint.Time) {
            if (!IsEqual(memStorePoint, tsdbPoint)) {
                inc(DifferentPoints_);

                if (hasDiffPoints) {
                    pointsDiff << ", ";
                }
                pointsDiff << "[" << memStorePoint.Time << "] " << memStorePoint << " != " << tsdbPoint;

                hasDiffPoints = true;
                isEqual = false;
            }
            memStoreHasNext = memStoreIter->Next(&memStorePoint);
            tsdbHasNext = tsdbIter->Next(&tsdbPoint);
        } else if (memStorePoint.Time < tsdbPoint.Time) {
            inc(TsdbMissingPoints_);
            isEqual = false;
            memStoreHasNext = memStoreIter->Next(&memStorePoint);
        } else {
            inc(MemStoreMissingPoints_);
            isEqual = false;
            tsdbHasNext = tsdbIter->Next(&tsdbPoint);
        }
    }

    while (memStoreHasNext) {
        inc(TsdbMissingPoints_);
        isEqual = false;
        memStoreHasNext = memStoreIter->Next(&memStorePoint);
    }

    while (tsdbHasNext) {
        inc(MemStoreMissingPoints_);
        isEqual = false;
        tsdbHasNext = tsdbIter->Next(&tsdbPoint);
    }

    if (hasDiffPoints) {
        inc(MetricsWithDiff_);
        MON_INFO(Comparison, "metric with different points: " << KeyToString(key, strings));
        MON_DEBUG(Comparison, "points diff: " << KeyToString(key, strings) << "; [MemStore] != [TSDB]: " << pointsDiff);
    }

    if (isEqual) {
        inc(EqualMetrics_);
    }
}

} // namespace NSolomon::NDataProxy
