#pragma once

#include <solomon/services/dataproxy/lib/cluster_id/dc.h>
#include <solomon/services/dataproxy/lib/datasource/comparison/storage_type.h>
#include <solomon/services/dataproxy/lib/datasource/query.h>

#include <library/cpp/containers/absl_flat_hash/flat_hash_map.h>
#include <library/cpp/monlib/metrics/metric_registry.h>

namespace NSolomon::NDataProxy {

enum class EAggregationType {
    Unknown,
    Host,
    Group
};

struct TMetricData {
    NMonitoring::EMetricType Type = NMonitoring::EMetricType::UNKNOWN;
    EAggregationType AggrType = EAggregationType::Unknown;
    EDc Dc = EDc::Unknown;

    std::unique_ptr<ITimeSeries> TimeSeries;

    static TMetricData FromMetric(TReadManyResult::TMetricData&& metric, const NStringPool::TStringPool& strings);
};

class TComparisonMetric {
public:
    TComparisonMetric(NMonitoring::TMetricRegistry& registry, NMonitoring::TLabels labels)
        : Registry_{registry}
        , OriginalLabels_{std::move(labels)}
    {
    }

    void Inc(const TMetricData& metric) {
        auto labels = OriginalLabels_;
        labels.Add("type", MetricTypeToStr(metric.Type));
        labels.Add("dc", DcToStr(metric.Dc));
        switch (metric.AggrType) {
            case EAggregationType::Unknown:
                labels.Add("aggregation", "unknown");
                break;
            case EAggregationType::Host:
                labels.Add("aggregation", "host");
                break;
            case EAggregationType::Group:
                labels.Add("aggregation", "group");
                break;
        }

        Registry_.Rate(labels)->Inc();
    }

private:
    NMonitoring::TMetricRegistry& Registry_;
    const NMonitoring::TLabels OriginalLabels_;
};

class TMetricComparator {
public:
    explicit TMetricComparator(NMonitoring::TMetricRegistry& registry)
        : MemStoreTotal_{registry, {{"sensor", "comparison.metrics.total"}, {"storage", "memstore"}}}
        , TsdbTotal_{registry, {{"sensor", "comparison.metrics.total"}, {"storage", "tsdb"}}}
        , MemStoreMissingMetrics_{registry, {{"sensor", "comparison.metrics.missing"}, {"storage", "memstore"}}}
        , TsdbMissingMetrics_{registry, {{"sensor", "comparison.metrics.missing"}, {"storage", "tsdb"}}}
        , MemStoreEmpty_{registry, {{"sensor", "comparison.metrics.empty"}, {"storage", "memstore"}}}
        , TsdbEmpty_{registry, {{"sensor", "comparison.metrics.empty"}, {"storage", "tsdb"}}}
        , TypeMismatch_{registry, {{"sensor", "comparison.metrics.type_mismatch"}}}
        , PresentMetrics_{registry, {{"sensor", "comparison.metrics.present"}}}
        , MemStoreMissingPoints_{registry, {{"sensor", "comparison.points.missing"}, {"storage", "memstore"}}}
        , TsdbMissingPoints_{registry, {{"sensor", "comparison.points.missing"}, {"storage", "tsdb"}}}
        , DifferentPoints_{registry, {{"sensor", "comparison.points.diff"}}}
        , MetricsWithDiff_{registry, {{"sensor", "comparison.metrics.with_diff"}}}
        , EqualMetrics_{registry, {{"sensor", "comparison.metrics.equal"}}}
    {
    }

    void Compare();

    std::unique_ptr<TReadManyResult> CopyResponse(TReadManyResult&& response, EStorageType type);
    void CaptureResponse(TReadManyResult&& response, EStorageType type);

    void AddError(const TDataSourceError& error);

private:
    using TMetricArray = std::array<TMetricData, ToUnderlying(EStorageType::Count)>;
    void Compare(const TMetricKey<ui32>& key, TMetricArray&& data, const NStringPool::TStringPool& strings);
    void ComparePoints(
            const TMetricKey<ui32>& key,
            TMetricData&& memStoreMetric,
            TMetricData&& tsdbMetric,
            const NStringPool::TStringPool& strings);

private:
    absl::flat_hash_map<TMetricKey<ui32>, TMetricArray> Metrics_;
    NStringPool::TStringPoolBuilder Strings_;

    TComparisonMetric MemStoreTotal_;
    TComparisonMetric TsdbTotal_;
    TComparisonMetric MemStoreMissingMetrics_;
    TComparisonMetric TsdbMissingMetrics_;
    TComparisonMetric MemStoreEmpty_;
    TComparisonMetric TsdbEmpty_;
    TComparisonMetric TypeMismatch_;
    TComparisonMetric PresentMetrics_;

    TComparisonMetric MemStoreMissingPoints_;
    TComparisonMetric TsdbMissingPoints_;
    TComparisonMetric DifferentPoints_;
    TComparisonMetric MetricsWithDiff_;
    TComparisonMetric EqualMetrics_;
};

} // namespace NSolomon::NDataProxy
