#include "comparison.h"
#include "comparator.h"
#include "storage_type.h"

#include <solomon/libs/cpp/actors/events/events.h>
#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/event_local.h>
#include <library/cpp/actors/core/hfunc.h>

#include <array>

using namespace NActors;
using namespace NSolomon::NTracing;
using yandex::monitoring::dataproxy::ShortTermStorage;

namespace NSolomon::NDataProxy {
namespace {

struct TPrivateEvent: private TPrivateEvents {
    enum {
        Response = SpaceBegin,
        Error,
        End
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TResponse: public TEventLocal<TResponse, Response> {
        TResponse(std::unique_ptr<TReadManyResult> response, EStorageType type)
            : Response(std::move(response))
            , StorageType(type)
        {
        }

        std::unique_ptr<TReadManyResult> Response;
        EStorageType StorageType;
    };

    struct TError: public TEventLocal<TError, Error> {
        TError(TString&& project, EDataSourceStatus status, TString&& message, EStorageType type)
            : Project(std::move(project))
            , Status(status)
            , Message(std::move(message))
            , StorageType(type)
        {
        }

        TString Project;
        EDataSourceStatus Status;
        TString Message;
        EStorageType StorageType;
    };
};

class TRequester: public TActor<TRequester> {
public:
    TRequester(IResultHandlerPtr<TReadManyResult> handler, NMonitoring::TMetricRegistry& registry, EStorageType mainStorage)
        : TActor<TRequester>{&TRequester::StateWork}
        , Handler_{std::move(handler)}
        , Comparator_{registry}
        , MainStorage_{mainStorage}
    {
    }

private:
    void OnResponse(TPrivateEvent::TResponse::TPtr& ev) {
        auto& response = ev->Get()->Response;
        auto type = ev->Get()->StorageType;

        if (type == MainStorage_) {
            Handler_->OnSuccess(Comparator_.CopyResponse(std::move(*response), type));
        } else {
            Comparator_.CaptureResponse(std::move(*response), type);
        }

        ++SuccessCount_;

        if (SuccessCount_ + ErrorCount_ == ToUnderlying(EStorageType::Count)) {
            Comparator_.Compare();
            PassAway();
        }
    }

    void OnError(TPrivateEvent::TError::TPtr& ev) {
        ++ErrorCount_;

        auto type = ev->Get()->StorageType;
        Comparator_.AddError(TDataSourceError{ev->Get()->Status, ev->Get()->Message});

        if (type == MainStorage_) {
            Handler_->OnError(std::move(ev->Get()->Project), ev->Get()->Status, std::move(ev->Get()->Message));
        }

        if (SuccessCount_ + ErrorCount_ == ToUnderlying(EStorageType::Count)) {
            Comparator_.Compare();
            PassAway();
        }
    }

    STATEFN(StateWork) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TPrivateEvent::TResponse, OnResponse);
            hFunc(TPrivateEvent::TError, OnError);
        }
    }

private:
    size_t SuccessCount_ = 0;
    size_t ErrorCount_ = 0;

    IResultHandlerPtr<TReadManyResult> Handler_;
    TMetricComparator Comparator_;
    EStorageType MainStorage_;
};

struct THandler: public IResultHandler<TReadManyResult> {
    THandler(const TActorSystem& sys, TActorId actor, EStorageType type)
        : Sys(sys)
        , Actor(actor)
        , Type(type)
    {
    }

    void OnSuccess(std::unique_ptr<TReadManyResult> response) override {
        Sys.Send(Actor, new TPrivateEvent::TResponse(std::move(response), Type));
    }

    void OnError(TString&& project, EDataSourceStatus status, TString&& message) override {
        Sys.Send(Actor, new TPrivateEvent::TError(std::move(project), status, std::move(message), Type));
    }

    const TActorSystem& Sys;
    TActorId Actor;
    EStorageType Type;
};

class TComparisonSource: public IDataSource {
public:
    TComparisonSource(
            TActorRuntime& runtime,
            IDataSourcePtr memStore,
            IDataSourcePtr tsdb,
            NMonitoring::TMetricRegistry& registry,
            EStorageType defaultStorage,
            bool doCompare)
        : Runtime_{runtime}
        , MemStore_{std::move(memStore)}
        , Tsdb_{std::move(tsdb)}
        , Registry_{registry}
        , DefaultStorage_{defaultStorage}
        , DoCompare_{doCompare}
    {
    }

    bool CanHandle(const TQuery& query) const override {
        switch (StorageType(query)) {
            case EStorageType::MemStore:
                return MemStore_->CanHandle(query);
            case EStorageType::Tsdb:
                return Tsdb_->CanHandle(query);
            case EStorageType::Count:
                Y_FAIL("Incorrect enum value");
        }
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::Find);
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::ResolveOne);
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::ResolveMany);
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::MetricNames);
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::LabelKeys);
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::LabelValues);
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::UniqueLabels);
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::ReadOne);
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        if (!DoCompare_ || !query.IsYasmSts()) {
            CallMethod(std::move(query), std::move(handler), std::move(traceCtx), &IDataSource::ReadMany);
            return;
        }

        auto mainStorage = StorageType(query);
        auto actor = Runtime_.Register(new TRequester{std::move(handler), Registry_, mainStorage});

        auto queryCopy = query.Clone();
        MemStore_->ReadMany(std::move(queryCopy), MakeIntrusive<THandler>(Runtime_.ActorSystem(), actor, EStorageType::MemStore));
        Tsdb_->ReadMany(std::move(query), MakeIntrusive<THandler>(Runtime_.ActorSystem(), actor, EStorageType::Tsdb));
    }

    void WaitUntilInitialized() const override {
        MemStore_->WaitUntilInitialized();
        Tsdb_->WaitUntilInitialized();
    }

private:
    EStorageType StorageType(const TQuery& query) const {
        switch (query.ShortTermStorage) {
            case ShortTermStorage::MEMSTORE:
                return EStorageType::MemStore;
            case ShortTermStorage::TSDB:
                return EStorageType::Tsdb;
            case ShortTermStorage::UNSPECIFIED:
                return query.IsYasmSts() ? DefaultStorage_ : EStorageType::MemStore;
            case ShortTermStorage::ShortTermStorage_INT_MAX_SENTINEL_DO_NOT_USE_:
            case ShortTermStorage::ShortTermStorage_INT_MIN_SENTINEL_DO_NOT_USE_:
                Y_FAIL("Incorrect enum value");
        }
    }

    template <typename TQuery, typename TResult>
    void CallMethod(
            TQuery&& query,
            IResultHandlerPtr<TResult>&& handler,
            TSpanId&& traceCtx,
            void (IDataSource::*method)(TQuery, IResultHandlerPtr<TResult>, TSpanId) const) const
    {
        switch (StorageType(query)) {
            case EStorageType::MemStore:
                std::invoke(method, MemStore_, std::move(query), std::move(handler), std::move(traceCtx));
                break;
            case EStorageType::Tsdb:
                std::invoke(method, Tsdb_, std::move(query), std::move(handler), std::move(traceCtx));
                break;
            case EStorageType::Count:
                Y_FAIL("Incorrect enum value");
        }
    }

private:
    TActorRuntime& Runtime_;
    IDataSourcePtr MemStore_;
    IDataSourcePtr Tsdb_;
    NMonitoring::TMetricRegistry& Registry_;
    EStorageType DefaultStorage_;
    bool DoCompare_;
};

} // namespace

IDataSourcePtr CreateComparisonSource(
        TActorRuntime& runtime,
        IDataSourcePtr memStore,
        IDataSourcePtr tsdb,
        NMonitoring::TMetricRegistry& registry,
        ShortTermStorage defaultSts,
        bool doCompare)
{
    EStorageType defaultStorage;
    switch (defaultSts) {
        case ShortTermStorage::MEMSTORE:
            defaultStorage = EStorageType::MemStore;
            break;
        case ShortTermStorage::TSDB:
            defaultStorage = EStorageType::Tsdb;
            break;
        case ShortTermStorage::UNSPECIFIED:
            Y_FAIL("Default short term storage must be specified");
        case ShortTermStorage::ShortTermStorage_INT_MAX_SENTINEL_DO_NOT_USE_:
        case ShortTermStorage::ShortTermStorage_INT_MIN_SENTINEL_DO_NOT_USE_:
            Y_FAIL("Incorrect enum value");
    }

    return MakeIntrusive<TComparisonSource>(runtime, std::move(memStore), std::move(tsdb), registry, defaultStorage, doCompare);
}

} // namespace NSolomon::NDataProxy
