#include "is_equal.h"

#include <library/cpp/monlib/metrics/log_histogram_snapshot.h>

namespace NSolomon::NDataProxy {
namespace {

static constexpr double EPS = 1e-9;

double Diff(double x, double y) {
    return std::abs(x - y);
}

bool IsEqualDouble(double lhs, double rhs) {
    return Diff(lhs, rhs) < EPS || (std::isnan(lhs) && std::isnan(rhs));
}

class TYasmHistogramComparator {
public:
    using TBucket = NTs::NValue::THistogram::TBucket;
    using TBuckets = std::vector<TBucket>;

    bool operator()(const TBuckets& lhs, const TBuckets& rhs) const {
        return Compare(lhs, rhs) || Compare(rhs, lhs);
    }

private:
    static bool Compare(const TBuckets& from, const TBuckets& to) {
        for (size_t i = 1; i + 1 < to.size(); ++i) {
            double boundBegin = to[i - 1].UpperBound;
            double boundEnd = to[i].UpperBound;

            auto end = std::lower_bound(from.begin(), from.end(), TBucket{boundEnd, 0u}, [](const auto& lhs, const auto& rhs) {
                return lhs.UpperBound < rhs.UpperBound;
            });

            auto begin = std::upper_bound(
                    from.begin(),
                    from.end(),
                    TBucket{boundBegin, 0u},
                    [](const auto& lhs, const auto& rhs) {
                        return lhs.UpperBound < rhs.UpperBound;
                    });

            if (begin == from.begin()) {
                if (to[i].Value == 0) {
                    continue;
                }

                return false;
            }

            double sum = 0.0;
            for (auto it = begin;; ++it) {
                sum += it->Value;

                if (it == end) {
                    break;
                }
            }

            double estimate = 0.0;
            if (boundBegin == (begin - 1)->UpperBound && boundEnd == end->UpperBound) {
                estimate = sum;
            } else {
                estimate = sum * (boundEnd - boundBegin) / (end->UpperBound - (begin - 1)->UpperBound);
            }

            if (Diff(estimate, static_cast<double>(to[i].Value)) > 1.0) {
                return false;
            }
        }

        return true;
    }
};

template <class TBucketProvider, class TPercentileCalculator>
double PercentileCompare(const TBucketProvider& lhs, const TBucketProvider& rhs, TPercentileCalculator calculator, double step) {
    double delta = step;
    double max = 0.0;
    while (delta < 1.0) {
        double lp = calculator(lhs, delta);
        double rp = calculator(rhs, delta);
        double d = Diff(lp, rp);
        if (d > max) {
            max = d;
        }
        delta += step;
    }

    return max;
}

template <class TBucketProvider>
double Percentile(const TBucketProvider& provider, double p) {
    if (provider.Count() == 0u) {
        return 0.0;
    }

    double sum = 0.0;

    for (size_t i = 0; i < provider.Count(); ++i) {
        sum += provider.Value(i);
    }

    if (sum == 0.0) {
        return 0.0;
    }

    double delta = 0.0;

    for (size_t i = 0; i < provider.Count(); ++i) {
        double pn = delta + provider.Value(i) / sum;

        if (pn >= p) {
            if (i == 0) {
                return 0.0;
            }

            double len = provider.UpperBound(i) - provider.UpperBound(i - 1);

            return provider.UpperBound(i - 1) + len * (p - delta) / (pn - delta);
        }

        delta = pn;
    }

    return 0.0;
}

template <class TBucketProvider>
double LogPercentile(const TBucketProvider& provider, double p) {
    double sum = 0.0;
    for (size_t i = 0; i < provider.Count(); ++i) {
        sum += provider.Value(i);
    }

    sum += provider.Zeros();

    double delta = provider.Zeros() / sum;
    if (delta >= p) {
        return 0.0;
    }

    for (size_t i = 0; i < provider.Count(); ++i) {
        delta += provider.Value(i) / sum;
        if (delta >= p) {
            return provider.UpperBound(i);
        }
    }

    return 0.0;
}

struct THistogramBucketProvider {
    THistogramBucketProvider(const NTs::NValue::THistogram& base)
        : Hist(base) {
    }

    double Value(size_t i) const {
        return Hist.Buckets[i].Value;
    }

    double UpperBound(size_t i) const {
        return Hist.Buckets[i].UpperBound;
    }

    size_t Count() const {
        Y_ENSURE(Hist.Buckets.size() >= 1u);
        return Hist.Buckets.size() - 1;
    }

    const NTs::NValue::THistogram& Hist;
};

struct TLogHistogramBucketProvider {
    ui64 Zeros() const {
        return Base.ZeroCount;
    }

    ui64 Count() const {
        return Base.Values.size();
    }

    double Value(size_t i) const {
        return Base.Values[i];
    }

    double UpperBound(size_t i) const {
        return std::pow(Base.Base, Base.StartPower + static_cast<int>(i));
    }

    const NTs::NValue::TLogHistogram& Base;
};

bool IsEqual(const NTs::NValue::TDouble& lhs, const NTs::NValue::TDouble& rhs) {
    return lhs.Denom == rhs.Denom && IsEqualDouble(lhs.Num, rhs.Num);
}

bool IsEqual(const NTs::NValue::TLong& lhs, const NTs::NValue::TLong& rhs) {
    return lhs == rhs;
}

bool IsEqual(const NTs::NValue::TSummaryInt& lhs, const NTs::NValue::TSummaryInt& rhs) {
    return lhs == rhs;
}

bool IsEqual(const NTs::NValue::TSummaryDouble& lhs, const NTs::NValue::TSummaryDouble& rhs) {
    return lhs.CountValue == rhs.CountValue &&
           IsEqualDouble(lhs.Last, rhs.Last) &&
           IsEqualDouble(lhs.Max, rhs.Max) &&
           IsEqualDouble(lhs.Min, rhs.Min) &&
           IsEqualDouble(lhs.Sum, rhs.Sum);
}

bool IsEqual(const NTs::NValue::THistogram& lhs, const NTs::NValue::THistogram& rhs) {
    THistogramBucketProvider providerLhs(lhs);
    THistogramBucketProvider providerRhs(rhs);
    TYasmHistogramComparator comparator;

    return PercentileCompare(providerLhs, providerRhs, Percentile<THistogramBucketProvider>, 0.05) < EPS ||
           comparator(lhs.Buckets, rhs.Buckets);
}

bool IsEqual(const NTs::NValue::TLogHistogram& lhs, const NTs::NValue::TLogHistogram& rhs) {
    TLogHistogramBucketProvider lhsP{lhs};
    TLogHistogramBucketProvider rhsP{rhs};

    return PercentileCompare(lhsP, rhsP, LogPercentile<TLogHistogramBucketProvider>, 0.05) < EPS;
}

} // namespace

bool IsEqual(const NTs::TVariantPoint& lhs, const NTs::TVariantPoint& rhs) {
    return std::visit([](auto&& lhs, auto&& rhs) {
        if constexpr (std::is_same_v<decltype(lhs), decltype(rhs)>) {
            return IsEqual(lhs, rhs);
        } else {
            return false;
        }
    }, lhs.Value, rhs.Value);
}

} // namespace NSolomon::NDataProxy
