#include "data_read_actors.h"
#include "errors.h"
#include "find_marshaller.h"
#include "resolve_many_marshaller.h"
#include "marshaller_helpers.h"

#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/stockpile/events.h>
#include <solomon/services/dataproxy/lib/timeseries/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>
#include <solomon/services/dataproxy/lib/timeseries/aggregates.h>
#include <solomon/services/dataproxy/lib/datasource/request_manager.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>
#include <library/cpp/containers/heap_dict/heap_dict.h>
#include <library/cpp/containers/absl_flat_hash/flat_hash_set.h>

#include <util/string/builder.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

struct TMetricMeta {
    std::shared_ptr<TMetricKey<ui32>> Key;
    NMonitoring::EMetricType Type;
    TStockpileId Id;
};

struct TMetricData {
    TMetricMeta* Meta{nullptr};
    TStockpileId Id;
    std::unique_ptr<ITimeSeries> TimeSeries;
    std::unique_ptr<TAggregateVariant> Aggregate;
};

struct TDataSourceStatus {
    EDataSourceStatus Status;
    TString Message;
};

enum class EMetaRequestState : ui8 {
    Skipped,
    Sent,
    Completed,
};

ui64 ToCookie(EReplica replica) {
    return static_cast<ui64>(replica);
}

EReplica FromCookie(ui64 cookie) {
    return static_cast<EReplica>(cookie);
}

using TMetricsMeta = std::vector<TMetricMeta>;
using TMetaErrors = std::vector<std::pair<TStringBuf, TDataSourceStatus>>;

template <typename TDerived>
struct TMergeStateBase {
    TMetaErrors Errors;

    void AddError(EReplica replica, TStringBuf replicaName, EDataSourceStatus status, TString&& message) {
        static_cast<TDerived*>(this)->ClearReplica(replica);
        Errors.emplace_back(replicaName, TDataSourceStatus{status, message});
    }

    [[nodiscard]] TDataSourceStatus MergeErrors() {
        EDataSourceStatus mergedStatus = EDataSourceStatus::NOT_FOUND;
        TStringBuilder reply;

        if (Errors.size() == 1) {
            return std::move(Errors[0].second);
        }

        reply << "request failed in all replicas: ";

        for (auto& [replicaName, replicaError]: Errors) {
            if (replicaError.Status != EDataSourceStatus::NOT_FOUND) {
                mergedStatus = replicaError.Status;
            }
            reply << '[' << replicaName << "] " << replicaError.Status << ": " <<  replicaError.Message;
        }
        return TDataSourceStatus{mergedStatus, std::move(reply)};
    }
};

template <typename T>
concept HasTotalCount = requires(T t) {
    t.totalcount();
};

struct TMetaMergeState: public TMergeStateBase<TMetaMergeState> {
    NStringPool::TStringPoolBuilder Strings;
    TReplicaMap<std::optional<TMetricsMeta>> ReplicaMetrics;
    ui32 MetricsLimit;

    template <typename TResponse>
    bool TryAddResponse(EReplica replica, const TResponse& resp) {
        auto& metrics = ReplicaMetrics[replica];
        if (!metrics.has_value()) {
            metrics.emplace(TMetricsMeta{});
        }
        metrics->reserve(metrics->size() + resp.metrics_size());
        if constexpr (HasTotalCount<TResponse>) {
            if ((ui32)resp.metrics_size() < resp.totalcount()) {
                return false;
            }
        }
        for (const auto& metric: resp.metrics()) {
            metrics->push_back(TMetricMeta{
                    std::make_shared<TMetricKey<ui32>>(MetricKeyFromProto(metric, &Strings)),
                    NSolomon::FromProto(metric.type()),
                    TStockpileId{metric.metric_id().shard_id(), metric.metric_id().local_id()}});
        }
        if (MetricsLimit != 0 && metrics->size() > MetricsLimit) {
            return false;
        }
        return true;
    }

    void ClearReplica(EReplica replica) {
        ReplicaMetrics[replica] = std::nullopt;
    }

    bool Empty() {
        for (auto replica: KnownReplicas) {
            if (ReplicaMetrics[replica].has_value()) {
                return false;
            }
        }
        return true;
    }
};

struct TMetricKeyPtrHash {
    size_t operator()(const std::shared_ptr<TMetricKey<ui32>>& mk) const {
        return mk->Hash();
    }
};

struct TMetricKeyPtrEq {
    bool operator()(const std::shared_ptr<TMetricKey<ui32>>& lhs, const std::shared_ptr<TMetricKey<ui32>>& rhs) const {
        return *lhs == *rhs;
    }
};

using TMetricsData = std::unordered_map<std::shared_ptr<TMetricKey<ui32>>, TReplicaMap<std::optional<TMetricData>>, TMetricKeyPtrHash, TMetricKeyPtrEq>;

struct TPrivateEvents: private NSolomon::TPrivateEvents {
    enum {
        Timeout = SpaceBegin,
        End
    };

    static_assert(End < SpaceEnd, "too many event types");

    struct TTimeout: public TEventLocal<TTimeout, Timeout> {
        const TMetricKey<ui32>* MetricKey;

        explicit TTimeout(const TMetricKey<ui32>* metricKey)
                : MetricKey{metricKey}
        {
        }
    };
};

struct TDataMergeState: public TMergeStateBase<TDataMergeState> {
    struct TShard {
        std::unordered_map<ui64, TMetricMeta*> IdToMeta;
    };

    struct TReplicaState {
        TMetricsMeta Meta;
        std::map<ui32, TShard> Shards;

        explicit TReplicaState(TMetricsMeta&& meta)
            : Meta{std::move(meta)}
        {
            for (auto& m: Meta) {
                TShard& shard = Shards[m.Id.ShardId];
                shard.IdToMeta[m.Id.LocalId] = &m;
            }
        }
    };

    NStringPool::TStringPoolBuilder Strings;
    TReplicaMap<std::optional<TReplicaState>> ReplicaState;
    TMetricsData MetricsData;

    explicit TDataMergeState(TMetaMergeState&& metaState)
        : Strings{std::move(metaState.Strings)}
    {
        for (auto replica: KnownReplicas) {
            if (metaState.ReplicaMetrics[replica].has_value()) {
                auto& metrics = metaState.ReplicaMetrics[replica].value();

                for (const auto& metric: metrics) {
                    MetricsData.emplace(metric.Key, TReplicaMap<std::optional<TMetricData>>{});
                }
                ReplicaState[replica].emplace(std::move(metrics));
            }
        }
    }

    void AddResponse(EReplica replica, yandex::solomon::stockpile::TCompressedReadManyResponse resp) {
        if (!ReplicaState[replica].has_value()) {
            return;
        }

        auto& replicaState = ReplicaState[replica].value();

        for (int i = 0; i < resp.metrics_size(); i++) {
            auto* metricProto = resp.mutable_metrics(i);

            ui32 shardId = metricProto->shardid();
            ui64 localId = metricProto->localid();

            auto shardIt = replicaState.Shards.find(shardId);
            if (shardIt == replicaState.Shards.end()) {
                continue;
            }

            auto metaIt = shardIt->second.IdToMeta.find(localId);
            if (metaIt == shardIt->second.IdToMeta.end()) {
                continue;
            }

            auto& metricPerReplica = MetricsData[metaIt->second->Key];

            if (!metricPerReplica[replica].has_value()) {
                metricPerReplica[replica].emplace(TMetricData{});
            }

            auto& metric = metricPerReplica[replica].value();
            metric.Meta = metaIt->second;
            metric.TimeSeries = FromProto(metric.Meta->Type, *metricProto);

            if (metricProto->aggregate_case() != yandex::solomon::stockpile::MetricData::AGGREGATE_NOT_SET) {
                metric.Aggregate = std::make_unique<TAggregateVariant>();
                FromProto(*metricProto, metric.Aggregate.get());
            }
        }
    }

    void ClearReplica(EReplica) {
        // no-op
    }

    bool Empty() {
        for (auto replica: KnownReplicas) {
            if (ReplicaState[replica].has_value()) {
                return false;
            }
        }
        return true;
    }
};

class TReadManyActor: public TActorBootstrapped<TReadManyActor> {
public:
    TReadManyActor(
            const TLtsReplicas& replicas,
            TReadManyQuery query,
            IResultHandlerPtr<TReadManyResult> handler,
            NTracing::TSpanId traceCtx)
        : Replicas_{replicas}
        , Query_{std::move(query)}
        , Handler_{std::move(handler)}
        , TraceCtx_{std::move(traceCtx)}
    {
    }

    STATEFN(QueryMetabase) {
        try {
            switch (ev->GetTypeRewrite()) {
                hFunc(TMetabaseEvents::TFindResp, OnFindResponse);
                hFunc(TMetabaseEvents::TResolveManyResp, OnResolveManyResponse);
                hFunc(TMetabaseEvents::TDone, OnMetabaseDone);
                hFunc(TMetabaseEvents::TError, OnMetabaseError);
                sFunc(TMetabaseEvents::TWakeupEvent, OnMetabaseWakeup);
            }
        } catch (...) {
            Handler_->OnError(std::move(Query_.Project), EDataSourceStatus::BAD_REQUEST, CurrentExceptionMessage());
            PassAway();
        }
    }

    STATEFN(QueryStockpile) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TStockpileEvents::TReadManyResp, OnStockpileResp);
            hFunc(TStockpileEvents::TError, OnStockpileError);
            hFunc(TPrivateEvents::TTimeout, OnTimeout);
        }
    }

    void Bootstrap() {
        Become(&TThis::QueryMetabase);
        MetaMergeState_ = std::make_unique<TMetaMergeState>();

        ReqManager_ = std::make_unique<TRequestsManager>(Lts, SelfId());

        if (Query_.ResolvedKeys) {
            SendResolveManyRequests();
        } else if (Query_.Lookup) {
            SendFindRequests();
        } else {
            Handler_->OnError(std::move(Query_.Project),
                    EDataSourceStatus::BAD_REQUEST,
                    "invalid read many query, selectors and resolved keys are empty");
            PassAway();
        }
    }

    void SendFindRequests() {
        TFindQuery query;
        query.Selectors = Query_.Lookup->Selectors;
        query.Limit = Query_.Lookup->Limit;
        query.Deadline = Query_.Deadline;
        query.Time = Query_.Time;
        query.Project = Query_.Project;
        query.ForceReplicaRead = Query_.ForceReplicaRead;

        MetaMergeState_->MetricsLimit = Query_.Lookup->Limit;

        // TODO: split marshaller to 1) request builder, 2) response collector
        TFindMarshaller marshaller{std::move(query)};

        for (auto replica: KnownReplicas) {
            // TODO: do not use dc names for forcing replica read
            const auto& maybeLtsReplica = Replicas_[replica];
            if (!maybeLtsReplica) {
                continue;
            }
            const TLtsReplica& ltsReplica = *maybeLtsReplica;
            if (!Query_.ForceReplicaRead || ltsReplica.Name() == Query_.ForceReplicaRead) {
                auto event = std::make_unique<TMetabaseEvents::TFindReq>();
                event->Deadline = Query_.Deadline;
                event->ShardSelector = marshaller.ShardSelector();

                auto msg = std::make_shared<TMetabaseEvents::TFindReq::TProtoMsg>();
                marshaller.FillRequest(msg.get());
                event->Message = std::move(msg);
                auto span = TRACING_NEW_SPAN_START(TraceCtx_, "Find in " << ltsReplica.ClusterId);
                Send(ltsReplica.MetabaseClusterId, event.release(), 0, ToCookie(replica), std::move(span));
                ++InFlightRequestsPerReplica_[replica];
                MetaRequestStatus_[replica] = EMetaRequestState::Sent;
            }
        }

        for (auto replica: KnownReplicas) {
            if (InFlightRequestsPerReplica_[replica] > 0) {
                return;
            }
        }

        // reply with an empty response
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::make_unique<TReadManyResult>());
        PassAway();
    }

    void OnFindResponse(TMetabaseEvents::TFindResp::TPtr& ev) {
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));
        try {
            if (!MetaMergeState_->TryAddResponse(replica, *(ev->Get()->Message))) {
                Handler_->OnError(std::move(Query_.Project),
                                  EDataSourceStatus::BAD_REQUEST,
                                  TStringBuilder{} << "too many metrics requested, limit is " << MetaMergeState_->MetricsLimit);
                PassAway();
            }
            MetricCount_ += ev->Get()->Message->metrics_size();
        } catch (...) {
            MetaMergeState_->AddError(replica, Replicas_[replica]->Name(), EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot handle metabase response, " << CurrentExceptionMessage());
        }
    }

    TVector<TResolveManyQuery> MakeResolveManyQueries() {
        auto& resolvedKeys = *Query_.ResolvedKeys;
        i64 clusterKey = -1;
        i64 clusterValue = -1;

        i64 serviceKey = -1;
        i64 serviceValue = -1;

        for (auto [key, value]: resolvedKeys.CommonLabels) {
            if (resolvedKeys.Strings[key] == NLabels::LABEL_CLUSTER) {
                clusterKey = key;
                clusterValue = value;
            } else if (resolvedKeys.Strings[key] == NLabels::LABEL_SERVICE) {
                serviceKey = key;
                serviceValue = value;
            }
        }

        // single-shard request
        if (clusterValue != -1 && serviceValue != -1) {
            TResolveManyQuery query;
            query.Project = Query_.Project;
            query.Time = Query_.Time;
            query.Deadline = Query_.Deadline;
            query.ForceReplicaRead = Query_.ForceReplicaRead;
            query.CommonLabels = std::move(resolvedKeys.CommonLabels);
            query.MetricKeys = std::move(resolvedKeys.MetricKeys);
            query.Strings = std::move(resolvedKeys.Strings);

            TVector<TResolveManyQuery> result;
            result.push_back(std::move(query));
            return result;
        }

        absl::flat_hash_map<std::pair<ui32, ui32>, TVector<TMetricKey<ui32>>> clusterAndServiceToMetricKeys;
        for (auto& metricKey: resolvedKeys.MetricKeys) {
            std::pair<ui32, ui32> clusterAndService;
            if (clusterValue != -1) {
                clusterAndService.first = clusterValue;
            }
            if (serviceValue != -1) {
                clusterAndService.second = serviceValue;
            }
            for (auto [key, value]: metricKey.Labels) {
                if (resolvedKeys.Strings[key] == NLabels::LABEL_CLUSTER) {
                    clusterKey = key;
                    clusterAndService.first = value;
                } else if (resolvedKeys.Strings[key] == NLabels::LABEL_SERVICE) {
                    serviceKey = key;
                    clusterAndService.second = value;
                }
            }
            clusterAndServiceToMetricKeys[clusterAndService].push_back(std::move(metricKey));
        }

        Y_ENSURE(clusterKey != -1 && serviceKey != -1, "can't find cluster or service label");

        TVector<TResolveManyQuery> result;
        for (auto&& [clusterAndService, metricKeys]: clusterAndServiceToMetricKeys) {
            for (auto& metricKey: metricKeys) {
                EraseIf(metricKey.Labels, [=](const auto& label) {return label.Key == serviceKey || label.Key == clusterKey; });
            }
            TResolveManyQuery query;
            query.Project = Query_.Project;
            query.Time = Query_.Time;
            query.Deadline = Query_.Deadline;
            query.ForceReplicaRead = Query_.ForceReplicaRead;
            std::tie(clusterValue, serviceValue) = clusterAndService;
            query.CommonLabels = {
                {static_cast<ui32>(clusterKey), static_cast<ui32>(clusterValue)},
                {static_cast<ui32>(serviceKey), static_cast<ui32>(serviceValue)}
            };
            query.MetricKeys = std::move(metricKeys);
            query.Strings = resolvedKeys.Strings.Copy();

            result.push_back(std::move(query));
        }
        return result;
    }

    void SendResolveManyRequests() {
        for (auto&& query: MakeResolveManyQueries()) {
            // TODO: split marshaller to 1) request builder, 2) response collector
            TResolveManyMarshaller marshaller{std::move(query)};

            for (auto replica: KnownReplicas) {
                const auto& maybeLtsReplica = Replicas_[replica];
                if (!maybeLtsReplica) {
                    continue;
                }
                const auto& ltsReplica = *maybeLtsReplica;
                if (!Query_.ForceReplicaRead || ltsReplica.Name() == Query_.ForceReplicaRead) {
                    auto event = std::make_unique<TMetabaseEvents::TResolveManyReq>();
                    event->Deadline = Query_.Deadline;
                    event->ShardSelector = marshaller.ShardSelector();

                    auto msg = std::make_shared<TMetabaseEvents::TResolveManyReq::TProtoMsg>();
                    marshaller.FillRequest(msg.get());
                    event->Message = std::move(msg);
                    auto span = TRACING_NEW_SPAN_START(TraceCtx_, "ReadMany in " << ltsReplica.ClusterId);
                    Send(ltsReplica.MetabaseClusterId, event.release(), 0, ToCookie(replica), std::move(span));
                    ++InFlightRequestsPerReplica_[replica];
                    MetaRequestStatus_[replica] = EMetaRequestState::Sent;
                }
            }
        }

        for (auto replica: KnownReplicas) {
            if (InFlightRequestsPerReplica_[replica] > 0) {
                return;
            }
        }

        // reply with an empty response
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::make_unique<TReadManyResult>());
        PassAway();
    }

    void OnResolveManyResponse(TMetabaseEvents::TResolveManyResp::TPtr& ev) {
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));
        try {
            if (!MetaMergeState_->TryAddResponse(replica, *(ev->Get()->Message))) {
                Handler_->OnError(std::move(Query_.Project),
                                  EDataSourceStatus::BAD_REQUEST,
                                  TStringBuilder{} << "too many metrics requested, limit is " << MetaMergeState_->MetricsLimit);
                PassAway();
            }
            MetricCount_ += ev->Get()->Message->metrics_size();
        } catch (...) {
            WereErrorsFromMetabaseReplica_[replica] = true;
            MetaMergeState_->AddError(replica, Replicas_[replica]->Name(), EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot handle metabase response, " << CurrentExceptionMessage());
        }
    }

    void OnMetabaseError(TMetabaseEvents::TError::TPtr& ev) {
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));
        if (WereErrorsFromMetabaseReplica_[replica]) {
            return;
        }

        WereErrorsFromMetabaseReplica_[replica] = true;

        auto status = ToDataSourceStatus(ev->Get()->RpcCode, ev->Get()->MetabaseCode);
        MetaMergeState_->AddError(replica, Replicas_[replica]->Name(), status,
                TStringBuilder{} << "metabase error: " << ev->Get()->Message);
    }

    void OnMetabaseDone(TMetabaseEvents::TDone::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));
        --InFlightRequestsPerReplica_[replica];
        if (InFlightRequestsPerReplica_[replica] == 0) {
            OnReplicaMetaCompleted(replica);
        }
    }

    void OnReplicaMetaCompleted(EReplica replica) {
        Y_VERIFY(MetaRequestStatus_[replica] == EMetaRequestState::Sent);
        MetaRequestStatus_[replica] = EMetaRequestState::Completed;

        for (auto replica: KnownReplicas) {
            if (InFlightRequestsPerReplica_[replica] > 0) {
                if (!WereErrorsFromMetabaseReplica_[replica] && !MetaMergeState_->Empty()) {
                    if (Query_.SoftDeadline) {
                        // by schedule implementation, if soft deadline has already expired,
                        // event will be scheduled instantly
                        this->Schedule(*Query_.SoftDeadline, new TMetabaseEvents::TWakeupEvent{});
                    } else {
                        // wait response from second replica no more than 5s
                        // TODO: cancel replica request
                        this->Schedule(DEFAULT_TIMEOUT, new TMetabaseEvents::TWakeupEvent{});
                    }
                } // else wait for successful responses from other replicas

                return;
            }
        }

        // If all metabase requests have failed it is time to reply with error
        if (MetaMergeState_->Empty()) {
            auto mergedError = MetaMergeState_->MergeErrors();
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(Query_.Project), mergedError.Status, std::move(mergedError.Message));
            PassAway();
            return;
        }

        SendStockpileRequests();
    }

    void OnMetabaseWakeup() {
        for (auto replica: KnownReplicas) {
            if (InFlightRequestsPerReplica_[replica] > 0) {
                SendStockpileRequests();
                return;
            }
        }
    }

    void FilterYasmDownsampling(yandex::solomon::math::Operation& operation) {
        if (operation.has_downsampling() && Query_.IsYasm() &&
                operation.downsampling().fill_option() != yandex::solomon::math::OperationDownsampling::PREVIOUS) {
            operation.mutable_downsampling()->set_fill_option(yandex::solomon::math::OperationDownsampling::NONE);
        }
    }

    void SendStockpileRequests() {
        Become(&TThis::QueryStockpile);

        DataMergeState_ = std::make_unique<TDataMergeState>(std::move(*MetaMergeState_));

        for (auto replica: KnownReplicas) {
            const auto& maybeLtsReplica = Replicas_[replica];
            if (!maybeLtsReplica) {
                continue;
            }
            const TLtsReplica& ltsReplica = *maybeLtsReplica;
            auto replicaState = DataMergeState_->ReplicaState[replica];
            if (!replicaState.has_value()) {
                continue;
            }

            for (auto& [shardId, shard]: replicaState->Shards) {
                auto event = std::make_unique<TStockpileEvents::TReadManyReq>();
                event->Deadline = Query_.Deadline;
                event->Message.set_deadline(Query_.Deadline.MilliSeconds());
                event->Message.set_binaryversion(Query_.MaxTimeSeriesFormat);
                event->Message.set_frommillis(Query_.Time.From.MilliSeconds());
                event->Message.set_tomillis(Query_.Time.To.MilliSeconds());
                event->Message.set_shardid(shardId);
                event->Message.set_producer(Query_.Producer);

                auto* operations = event->Message.mutable_operations();
                operations->Reserve(Query_.Operations.size());
                for (const auto& operation: Query_.Operations) {
                    operations->Add()->CopyFrom(operation);
                    auto& addedOperation = operations->at(operations->size() - 1);
                    FilterYasmDownsampling(addedOperation);
                }

                auto* localIdsProto = event->Message.mutable_localids();
                localIdsProto->Reserve(shard.IdToMeta.size());

                TMetricKeys metricKeys;
                metricKeys.reserve(shard.IdToMeta.size());

                for (const auto& [localId, meta]: shard.IdToMeta) {
                    *localIdsProto->AddAlreadyReserved() = localId;
                    metricKeys.push_back(meta->Key.get());
                }

                auto reqId = ReqManager_->NewRequest(replica, std::move(metricKeys));

                auto span = TRACING_NEW_SPAN_START(TraceCtx_, "StockpileReadMany in " << ltsReplica.ClusterId << "/" << shardId);
                Send(ltsReplica.StockpileClusterId, event.release(), 0, reqId, std::move(span));
            }
        }

        if (!ReqManager_->HasInflightRequests()) {
            // Empty metabase response
            auto result = std::make_unique<TReadManyResult>();
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnSuccess(std::move(result));
            PassAway();
        }
    }

    void OnStockpileResp(TStockpileEvents::TReadManyResp::TPtr& ev) {
        TRACING_SPAN_END(ev->TraceId);

        ui64 reqId = ev->Cookie;

        bool error = false;
        auto replica = ReqManager_->GetReplica(reqId);

        try {
            DataMergeState_->AddResponse(replica, std::move(ev->Get()->Message));
        } catch (...) {
            error = true;
            DataMergeState_->AddError(replica, Replicas_[replica]->Name(), EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot handle stockpile response, " << CurrentExceptionMessage());
        }

        if (error) {
            ReqManager_->ReportError(reqId);
        } else {
            ReqManager_->ReportResponse(reqId, Query_.SoftDeadline, [this](TInstant deadline, const TMetricKey<ui32>* metricKey) {
                this->Schedule(deadline, new TPrivateEvents::TTimeout{metricKey});
            });
        }

        if (ReqManager_->IsCompleted()) {
            OnCompleted();
        }
    }

    void OnStockpileError(TStockpileEvents::TError::TPtr& ev) {
        TRACING_SPAN_END(ev->TraceId);

        auto status = ToDataSourceStatus(ev->Get()->RpcCode, ev->Get()->StockpileCode);
        auto replica = ReqManager_->GetReplica(ev->Cookie);
        ReqManager_->ReportError(ev->Cookie);

        DataMergeState_->AddError(replica, Replicas_[replica]->Name(), status,
                TStringBuilder{} << "stockpile error: " << ev->Get()->Message);

        if (ReqManager_->IsCompleted()) {
            OnCompleted();
        }
    }

    void OnTimeout(TPrivateEvents::TTimeout::TPtr& ev) {
        ReqManager_->ReportTimeout(ev->Get()->MetricKey);

        if (ReqManager_->IsCompleted()) {
            OnCompleted();
        }
    }

    void OnCompleted() {
        try {
            MergeAndReply();
        } catch(...) {
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(Query_.Project),
                    EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot merge metrics data, " << CurrentExceptionMessage());
            PassAway();
        }
    }

    void MergeAndReply() {
        auto result = std::make_unique<TReadManyResult>();
        result->Strings = std::move(DataMergeState_->Strings);
        result->Metrics.reserve(DataMergeState_->MetricsData.size());

        for (auto& [_, data]: DataMergeState_->MetricsData) {
            TVector<EReplica> mask;

            for (auto replica: KnownReplicas) {
                if (data[replica].has_value()) {
                    mask.push_back(replica);
                }
            }

            if (mask.empty()) {
                TDataSourceStatus mergedErrors = DataMergeState_->MergeErrors();
                TRACING_SPAN_END(TraceCtx_);
                Handler_->OnError(std::move(Query_.Project), mergedErrors.Status, std::move(mergedErrors.Message));
                PassAway();
                return;
            }

            result->Metrics.emplace_back(MergeMetricData(mask, std::move(data)));
        }

        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::move(result));
        PassAway();
    }

    static TReadManyResult::TMetricData MergeMetricData(TVector<EReplica>& mask, TReplicaMap<std::optional<TMetricData>>&& data) {
        TReadManyResult::TMetricData result;

        TMetricData& firstValidMetricData = data[mask[0]].value();

        result.Meta.Type = firstValidMetricData.Meta->Type;
        result.Meta.Name = firstValidMetricData.Meta->Key->Name;
        result.Meta.Labels = firstValidMetricData.Meta->Key->Labels; // TODO: move to avoid vector copy (be aware because Meta->Key is used as key in hash map)

        if (mask.size() == 1) {
            result.TimeSeries = std::move(firstValidMetricData.TimeSeries);
            result.Aggregate = std::move(firstValidMetricData.Aggregate);
            result.Meta.StockpileIds[mask[0]] = firstValidMetricData.Meta->Id;
            return result;
        }

        // (1) merge time series
        {
            TMergingIterator<TMaxMerger> it;
            NMonitoring::EMetricType metricDataType = NMonitoring::EMetricType::UNKNOWN;
            for (auto replica: KnownReplicas) {
                auto& metric = data[replica].value();
                result.Meta.StockpileIds[replica] = metric.Id;

                if (metric.TimeSeries && metric.TimeSeries->PointCount() > 0) {
                    it.AddTimeSeries(*metric.TimeSeries);
                    if (metricDataType == NMonitoring::EMetricType::UNKNOWN) {
                        metricDataType = metric.TimeSeries->Type();
                    }
                }
            }
            // TODO: do not recompress single non empty time series
            result.TimeSeries = std::make_unique<TCompressedTimeSeries>(metricDataType, it.Columns(), it);
        }

        // (2) find aggr with max count
        {
            auto maxIt = mask.end();
            for (auto it = mask.begin(); it != mask.end(); ++it) {
                auto& m = data[*it].value();
                if (m.Aggregate) {
                    if (maxIt == mask.end() || data[*maxIt]->Aggregate->Count < m.Aggregate->Count) {
                        maxIt = it;
                    }
                }
            }
            if (maxIt != mask.end()) {
                result.Aggregate = std::move(data[*maxIt]->Aggregate);
            }
        }

        return result;
    }

    static constexpr char ActorName[] = "Lts/ReadMany";

private:
    const TLtsReplicas& Replicas_;
    TReadManyQuery Query_;
    IResultHandlerPtr<TReadManyResult> Handler_;

    size_t MetricCount_;
    std::optional<size_t> OpTopIdx_;

    std::unique_ptr<TMetaMergeState> MetaMergeState_;
    std::unique_ptr<TDataMergeState> DataMergeState_;

    TReplicaMap<ui32> InFlightRequestsPerReplica_{0};
    TReplicaMap<bool> WereErrorsFromMetabaseReplica_{false};
    TReplicaMap<EMetaRequestState> MetaRequestStatus_{EMetaRequestState::Skipped};
    std::unique_ptr<TRequestsManager> ReqManager_;

    TSpanId TraceCtx_;
};

} // namespace

std::unique_ptr<IActor> ReadManyActor(
        const TLtsReplicas& replicas,
        TReadManyQuery query,
        IResultHandlerPtr<TReadManyResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TReadManyActor>(replicas, std::move(query), std::move(handler), std::move(traceCtx));
}

} // namespace NSolomon::NDataProxy
