#include "data_read_actors.h"
#include "errors.h"
#include "marshaller_helpers.h"

#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/stockpile/events.h>
#include <solomon/services/dataproxy/lib/timeseries/merger.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>

#include <solomon/libs/cpp/proto_convert/metric_type.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

#include <util/string/builder.h>

using namespace NActors;
using namespace NSolomon::NTracing;

namespace NSolomon::NDataProxy {
namespace {

ui64 ToCookie(EReplica replica) {
    return static_cast<ui64>(replica);
}

EReplica FromCookie(ui64 cookie) {
    return static_cast<EReplica>(cookie);
}

class TReadOneActor: public TActorBootstrapped<TReadOneActor> {
public:
    TReadOneActor(
            const TLtsReplicas& replicas,
            TReadOneQuery query,
            IResultHandlerPtr<TReadOneResult> handler,
            NTracing::TSpanId traceCtx)
        : Replicas_{replicas}
        , Query_{std::move(query)}
        , Handler_{std::move(handler)}
        , TraceCtx_{std::move(traceCtx)}
    {
        Y_UNUSED(TraceCtx_);
    }

    STATEFN(QueryMetabase) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TMetabaseEvents::TResolveOneResp, OnMetabaseResp);
            hFunc(TMetabaseEvents::TError, OnMetabaseError);
            hFunc(TMetabaseEvents::TDone, OnMetabaseDone);
        }
    }

    STATEFN(QueryStockpile) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TStockpileEvents::TReadOneResp, OnStockpileResp);
            hFunc(TStockpileEvents::TError, OnStockpileError);
        }
    }

    void Bootstrap() {
        SendMetabaseRequests();
    }

    void SendMetabaseRequests() {
        Become(&TThis::QueryMetabase);

        auto shardSelector = TShardSelector::FromLabels(Query_.Project, Query_.Labels);

        for (auto replica: KnownReplicas) {
            auto& maybeLtsReplica = Replicas_[replica];
            if (!maybeLtsReplica) {
                continue;
            }
            auto& ltsReplica = *maybeLtsReplica;
            auto event = std::make_unique<TMetabaseEvents::TResolveOneReq>();
            event->Deadline = Query_.Deadline;
            event->ShardSelector = shardSelector;

            auto msg = std::make_shared<TMetabaseEvents::TResolveOneReq::TProtoMsg>();

            msg->SetName(Query_.Name);
            LabelsToProto(Query_.Labels, msg->mutable_labels());
            msg->SetDeadlineMillis(Query_.Deadline.MilliSeconds());

            event->Message = std::move(msg);

            auto span = TRACING_NEW_SPAN_START(TraceCtx_, "ResolveOne in " << ltsReplica.ClusterId);
            Send(ltsReplica.MetabaseClusterId, event.release(), 0, ToCookie(replica), std::move(span));
            ++InFlightRequests_;
        }
    }

    void OnMetabaseResp(TMetabaseEvents::TResolveOneResp::TPtr& ev) {
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));

        try {
            auto& metric = ev->Get()->Message->metric();
            Metric_.Type = NSolomon::FromProto(metric.type());
            Metric_.Name = metric.name();
            LabelsFromProto(metric.labels(), &Metric_.Labels);

            TStockpileId& stockpileId = Metric_.StockpileIds[replica];
            stockpileId.ShardId = metric.metric_id().shard_id();
            stockpileId.LocalId = metric.metric_id().local_id();

            if (--InFlightRequests_ == 0) {
                SendStockpileRequests();
            }
        } catch (...) {
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(Query_.Project),
                    EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot handle metabase response, " << CurrentExceptionMessage());
            PassAway();
        }
    }

    void OnMetabaseError(TMetabaseEvents::TError::TPtr& ev) {
        auto status = ToDataSourceStatus(ev->Get()->RpcCode, ev->Get()->MetabaseCode);
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnError(std::move(Query_.Project), status, TStringBuilder{} << "metabase error: " << ev->Get()->Message);
        PassAway();
    }

    void OnMetabaseDone(TMetabaseEvents::TDone::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));
    }

    void SendStockpileRequests() {
        Become(&TThis::QueryStockpile);


        for (auto replica: KnownReplicas) {
            auto& maybeLtsReplica = Replicas_[replica];
            if (!maybeLtsReplica) {
                continue;
            }
            auto& ltsReplica = *maybeLtsReplica;
            auto event = std::make_unique<TStockpileEvents::TReadOneReq>();
            event->Deadline = Query_.Deadline;
            event->Message.set_deadline(Query_.Deadline.MilliSeconds());
            event->Message.set_binaryversion(Query_.MaxTimeseriesFormat);

            auto* metricId = event->Message.mutable_metric_id();
            metricId->set_shard_id(Metric_.StockpileIds[replica].ShardId);
            metricId->set_local_id(Metric_.StockpileIds[replica].LocalId);

            event->Message.set_frommillis(Query_.Time.From.MilliSeconds());
            event->Message.set_tomillis(Query_.Time.To.MilliSeconds());
            event->Message.set_producer(Query_.Producer);

            for (const auto& operation: Query_.Operations) {
                if (operation.has_downsampling()) {
                    const auto& downsampling = operation.downsampling();
                    event->Message.set_gridmillis(downsampling.grid_millis());
                    event->Message.set_aggregation(downsampling.aggregation());
                }
            }

            auto span = TRACING_NEW_SPAN_START(TraceCtx_, "ResolveOne in " << ltsReplica.ClusterId);
            Send(ltsReplica.StockpileClusterId, event.release(), 0, ToCookie(replica), std::move(span));
            ++InFlightRequests_;
        }
    }

    void OnStockpileResp(TStockpileEvents::TReadOneResp::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto replica = FromCookie(ev->Cookie);
        Y_VERIFY(IsKnownReplica(replica));

        try {
            TimeseriesPerReplica_[replica] = FromProto(Metric_.Type, ev->Get()->Message);

            if (--InFlightRequests_ == 0) {
                MergeAndReply();
            }
        } catch (...) {
            Handler_->OnError(std::move(Query_.Project),
                    EDataSourceStatus::UNKNOWN,
                    TStringBuilder{} << "cannot handle stockpile response, " << CurrentExceptionMessage());
            PassAway();
        }
    }

    void OnStockpileError(TStockpileEvents::TError::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        auto status = ToDataSourceStatus(ev->Get()->RpcCode, ev->Get()->StockpileCode);
        Handler_->OnError(std::move(Query_.Project), status, TStringBuilder{} << "stockpile error: " << ev->Get()->Message);
        PassAway();
    }

    void MergeAndReply() {
        auto result = std::make_unique<TReadOneResult>();
        result->Metric = std::move(Metric_);
        TVector<std::unique_ptr<ITimeSeries>> TimeseriesList;
        for (auto replica: KnownReplicas) {
            auto& timeseries = TimeseriesPerReplica_[replica];
            if (timeseries) {
                TimeseriesList.emplace_back(std::move(timeseries));
            }
        }
        result->TimeSeries = MergeTimeSeries(Metric_.Type, std::move(TimeseriesList));
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(std::move(result));
        PassAway();
    }

private:
    const TLtsReplicas& Replicas_;
    TReadOneQuery Query_;
    IResultHandlerPtr<TReadOneResult> Handler_;
    TMetric<TString> Metric_;
    TReplicaMap<std::unique_ptr<ITimeSeries>> TimeseriesPerReplica_;
    ui32 InFlightRequests_{0};
    NTracing::TSpanId TraceCtx_;
};

} // namespace

std::unique_ptr<IActor> ReadOneActor(
        const TLtsReplicas& replicas,
        TReadOneQuery query,
        IResultHandlerPtr<TReadOneResult> handler,
        TSpanId traceCtx)
{
    return std::make_unique<TReadOneActor>(replicas, std::move(query), std::move(handler), std::move(traceCtx));
}
} // namespace NSolomon::NDataProxy
