#include "errors.h"

using yandex::solomon::metabase::EMetabaseStatusCode;
using yandex::solomon::stockpile::EStockpileStatusCode;

namespace NSolomon::NDataProxy {
namespace {

EDataSourceStatus MapTransportError(const grpc::StatusCode &rpc) noexcept {
    // handle only transport errors
    switch (rpc) {
        case grpc::StatusCode::DEADLINE_EXCEEDED:
            return EDataSourceStatus::BACKEND_TIMEOUT;

        case grpc::StatusCode::UNAVAILABLE:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;

        default:
            return EDataSourceStatus::UNKNOWN;
    }
}

} // namespace

EDataSourceStatus ToDataSourceStatus(grpc::StatusCode rpc, EMetabaseStatusCode metabase) noexcept {
    if (auto status = MapTransportError(rpc); status != EDataSourceStatus::UNKNOWN) {
        return status;
    }

    switch (metabase) {
        case EMetabaseStatusCode::OK:
            return EDataSourceStatus::OK;

        case EMetabaseStatusCode::NOT_FOUND:
        case EMetabaseStatusCode::SHARD_NOT_FOUND:
            return EDataSourceStatus::NOT_FOUND;

        case EMetabaseStatusCode::INVALID_REQUEST:
        case EMetabaseStatusCode::SHARD_WRITE_ONLY:
            return EDataSourceStatus::BAD_REQUEST;

        case EMetabaseStatusCode::DEADLINE_EXCEEDED:
            return EDataSourceStatus::BACKEND_TIMEOUT;

        case EMetabaseStatusCode::INTERNAL_ERROR:
            return EDataSourceStatus::BACKEND_ERROR;

        case EMetabaseStatusCode::RESOURCE_EXHAUSTED:
        case EMetabaseStatusCode::NODE_UNAVAILABLE:
        case EMetabaseStatusCode::SHARD_NOT_READY:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;

        default:
            return EDataSourceStatus::UNKNOWN;
    }
}

EDataSourceStatus ToDataSourceStatus(grpc::StatusCode rpc, EStockpileStatusCode stockpile) noexcept {
    if (auto status = MapTransportError(rpc); status != EDataSourceStatus::UNKNOWN) {
        return status;
    }

    switch (stockpile) {
        case EStockpileStatusCode::OK:
            return EDataSourceStatus::OK;

        case EStockpileStatusCode::METRIC_NOT_FOUND:
        case EStockpileStatusCode::SHARD_ABSENT_ON_HOST:
            return EDataSourceStatus::NOT_FOUND;

        case EStockpileStatusCode::INVALID_REQUEST:
        case EStockpileStatusCode::UNSUPPORTED_BINARY_FORMAT:
            return EDataSourceStatus::BAD_REQUEST;

        case EStockpileStatusCode::DEADLINE_EXCEEDED:
            return EDataSourceStatus::BACKEND_TIMEOUT;

        case EStockpileStatusCode::INTERNAL_ERROR:
            return EDataSourceStatus::BACKEND_ERROR;

        case EStockpileStatusCode::RESOURCE_EXHAUSTED:
        case EStockpileStatusCode::NODE_UNAVAILABLE:
        case EStockpileStatusCode::SHARD_NOT_READY:
            return EDataSourceStatus::BACKEND_UNAVAILABLE;

        default:
            return EDataSourceStatus::UNKNOWN;
    }
}

} // namespace NSolomon::NDataProxy
