#include "find_marshaller.h"
#include "marshaller_helpers.h"

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/protos/metabase/grpc_find.pb.h>
#include <util/generic/algorithm.h>

namespace NSolomon::NDataProxy {

using yandex::solomon::metabase::FindRequest;
using yandex::solomon::metabase::FindResponse;

TFindMarshaller::TFindMarshaller(TFindQuery query) noexcept
    : Query_(std::move(query))
{
}

TShardSelector TFindMarshaller::ShardSelector() const {
    return TShardSelector::FromSelectors(Query_.Project, Query_.Selectors);
}

void TFindMarshaller::FillRequest(FindRequest* req) const {
    SelectorsToProto(Query_.Project, Query_.Selectors, req->mutable_selectors());
    req->MutableSliceOptions()->SetLimit(Query_.Limit);
    req->SetDeadlineMillis(Query_.Deadline.MilliSeconds());
    req->SetFillMetricName(Query_.FillMetricName);
}

void TFindMarshaller::AddResponse(EReplica replica, EDc dc, const FindResponse& resp) {
    Dcs_[replica] = dc;
    if (Metrics_.empty()) {
        Metrics_.reserve(Min(static_cast<ui32>(resp.metrics_size()), Query_.Limit));
    }

    auto updateState = [replica](const yandex::solomon::metabase::Metric& metric, TMetricState* state) {
        state->Type = NSolomon::FromProto(metric.type());
        TStockpileId& stockpileId = state->StockpileIds[replica];
        stockpileId.ShardId = metric.metric_id().shard_id();
        stockpileId.LocalId = metric.metric_id().local_id();
    };

    for (const auto& metric: resp.metrics()) {
        auto metricKey = MetricKeyFromProto(metric, &Strings_);

        if (Metrics_.size() < Query_.Limit) {
            auto [it, _] = Metrics_.emplace(std::move(metricKey), TMetricState{});
            updateState(metric, &it->second);
        } else {
            // do not add new metrics over the defined limit,
            // but here we can get response from another cluster which we have
            // to collect for adding stockpile ids
            if (auto it = Metrics_.find(metricKey); it != Metrics_.end()) {
                updateState(metric, &it->second);
            }
        }
    }

    TotalCount_ = Max(resp.GetTotalCount(), TotalCount_);
}

std::unique_ptr<TFindResult> TFindMarshaller::MakeResult() {
    auto result = std::make_unique<TFindResult>();
    result->TotalCount = TotalCount_;
    result->Strings = std::move(Strings_);

    result->Metrics.reserve(Min<size_t>(Query_.Limit, Metrics_.size()));
    for (auto& [key, state]: Metrics_) {
        if (result->Metrics.size() >= Query_.Limit) {
            break;
        }

        result->Metrics.emplace_back();
        auto& metric = result->Metrics.back();

        metric.Name = key.Name;
        metric.Labels = std::move(key.Labels); // XXX: move probably not working here

        metric.Type = state.Type;
        metric.StockpileIds = state.StockpileIds;
    }

    result->Dcs = Dcs_;

    return result;
}

} // namespace NSolomon::NDataProxy
