#include "label_keys_marshaller.h"
#include "marshaller_helpers.h"

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/protos/metabase/grpc_label_names.pb.h>

namespace NSolomon::NDataProxy {

using yandex::solomon::metabase::TLabelNamesRequest;
using yandex::solomon::metabase::TLabelNamesResponse;

TLabelKeysMarshaller::TLabelKeysMarshaller(TLabelKeysQuery query) noexcept
    : Query_(std::move(query))
{
}

TShardSelector TLabelKeysMarshaller::ShardSelector() const {
    return TShardSelector::FromSelectors(Query_.Project, Query_.Selectors);
}

void TLabelKeysMarshaller::FillRequest(TLabelNamesRequest* req) const {
    SelectorsToProto(Query_.Project, Query_.Selectors, req->mutable_selectors());
    req->set_deadlinemillis(Query_.Deadline.MilliSeconds());
}

void TLabelKeysMarshaller::AddResponse(EReplica, EDc, const TLabelNamesResponse& resp) {
    if (Keys_.empty()) {
        Keys_.reserve(resp.names_size());
    }

    for (const auto& key: resp.names()) {
        if (key != NYasm::AGGREGATED_MARKER) { // for viewing YASM data SOLOMON-7804
            AddKey(key);
        }
    }
}

std::unique_ptr<TLabelKeysResult> TLabelKeysMarshaller::MakeResult() {
    // because we quering Metabase per shard, we will never get
    // this special label keys and have to add them manually

    auto isYasmProject = NYasm::IsYasmProject(Query_.Project);

    if (!isYasmProject && !Query_.Selectors.Has(NLabels::LABEL_CLUSTER)) {
        AddKey(NLabels::LABEL_CLUSTER);
    }

    if (!isYasmProject && !Query_.Selectors.Has(NLabels::LABEL_SERVICE)) {
        AddKey(NLabels::LABEL_SERVICE);
    }

    auto result = std::make_unique<TLabelKeysResult>();
    result->Strings = std::move(Strings_);
    result->Keys = std::move(Keys_);
    return result;
}

void TLabelKeysMarshaller::AddKey(TStringBuf key) {
    ui32 keyId = Strings_.Put(key);

    // because of NStringPool::TStringPoolBuilder uses monotonically incremented ids
    // this is cheapest way to avoid names duplication
    if (static_cast<i32>(keyId) > LastKeyId_) {
        LastKeyId_ = static_cast<i32>(keyId);
        Keys_.push_back(keyId);
    }
}

} // namespace NSolomon::NDataProxy
