#include "label_values_marshaller.h"
#include "marshaller_helpers.h"

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/libs/cpp/yasm/constants/labels.h>
#include <solomon/libs/cpp/yasm/labels/non_yasm.h>
#include <solomon/protos/metabase/grpc_label_values.pb.h>

namespace NSolomon::NDataProxy {

using yandex::solomon::metabase::TLabelValuesRequest;
using yandex::solomon::metabase::TLabelValuesResponse;

TLabelValuesMarshaller::TLabelValuesMarshaller(TLabelValuesQuery query) noexcept
    : Query_(std::move(query))
{
}

TShardSelector TLabelValuesMarshaller::ShardSelector() const {
    return TShardSelector::FromSelectors(Query_.Project, Query_.Selectors);
}

void TLabelValuesMarshaller::FillRequest(TLabelValuesRequest* req) const {
    SelectorsToProto(Query_.Project, Query_.Selectors, req->mutable_selectors());
    for (const auto& key: Query_.Keys) {
        req->add_labels(key);
    }
    req->set_limit(Query_.Limit);
    req->set_textsearch(Query_.TextFilter);
    req->set_deadlinemillis(Query_.Deadline.MilliSeconds());
}

void TLabelValuesMarshaller::AddResponse(EReplica replica, EDc, const TLabelValuesResponse& resp) {
    if (Labels_.empty()) {
        Labels_.reserve(resp.values_size());
    }

    for (const auto& labelProto: resp.values()) {
        ui32 keyId = Strings_.Put(labelProto.name());

        auto& labelState = Labels_[keyId];
        labelState.Truncated = labelState.Truncated | labelProto.truncated();
        labelState.MetricCount = Max(labelState.MetricCount, labelProto.metric_count());

        if (labelState.Values.empty()) {
            labelState.Values.reserve(labelProto.values_size());
        }

        for (const auto& value: labelProto.values()) {
            if (labelState.Values.size() >= Query_.Limit) {
                break;
            }
            labelState.Values.insert(Strings_.Put(value));
        }
    }

    MetricCounts_[replica] += resp.metric_count();
}

std::unique_ptr<TLabelValuesResult> TLabelValuesMarshaller::MakeResult() {
    auto result = std::make_unique<TLabelValuesResult>();
    result->Project = Query_.Project;
    result->Strings = std::move(Strings_);

    result->Labels.reserve(Labels_.size());
    for (auto& [keyId, state]: Labels_) {
        result->Labels.emplace_back();
        auto& label = result->Labels.back();

        label.Values.reserve(state.Values.size());
        std::copy(std::begin(state.Values), std::end(state.Values), std::back_inserter(label.Values));

        label.Key = keyId;
        label.MetricCount = state.MetricCount;
        label.Truncated = state.Truncated;
    }

    for (auto replica: KnownReplicas) {
        result->MetricCount = Max(result->MetricCount, MetricCounts_[replica]);
    }

    return result;
}

} // namespace NSolomon::NDataProxy
