#include "data_read_actors.h"
#include "errors.h"
#include "find_marshaller.h"
#include "label_keys_marshaller.h"
#include "label_values_marshaller.h"
#include "lts.h"
#include "metric_names_marshaller.h"
#include "replica.h"
#include "resolve_many_marshaller.h"
#include "resolve_one_marshaller.h"
#include "unique_labels_marshaller.h"
#include "meta_merging_actor.h"

#include <solomon/services/dataproxy/config/datasource_config.pb.h>
#include <solomon/services/dataproxy/lib/cluster_map/cluster_map.h>
#include <solomon/services/dataproxy/lib/datasource/datasource.h>
#include <solomon/services/dataproxy/lib/datasource/lts/selfmon/lts_page.h>
#include <solomon/services/dataproxy/lib/initialization/events.h>
#include <solomon/services/dataproxy/lib/limits.h>
#include <solomon/services/dataproxy/lib/message_cache/cache_actor.h>
#include <solomon/services/dataproxy/lib/metabase/cluster.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/metabase/shard_actor.h>
#include <solomon/services/dataproxy/lib/stockpile/cluster.h>

#include <solomon/libs/cpp/actors/runtime/actor_runtime.h>
#include <solomon/libs/cpp/actors/scheduler/scheduler.h>
#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/selfmon/selfmon.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

using namespace NActors;
using namespace NSolomon::NTracing;
using yandex::solomon::config::rpc::TGrpcClientConfig;

namespace NSolomon::NDataProxy {
namespace {

struct TCacheResponses : public IResponseFactory {
    std::shared_ptr<google::protobuf::Message> Make(const TString& eventType) const override {
        if (eventType == "FindResponse") {
            return std::make_shared<yandex::solomon::metabase::FindResponse>();
        } else if (eventType == "ResolveOneResponse") {
            return std::make_shared<yandex::solomon::metabase::ResolveOneResponse>();
        } else if (eventType == "ResolveManyResponse") {
            return std::make_shared<yandex::solomon::metabase::ResolveManyResponse>();
        } else if (eventType == "MetricNamesResponse") {
            return std::make_shared<yandex::solomon::metabase::MetricNamesResponse>();
        } else if (eventType == "TLabelValuesResponse") {
            return std::make_shared<yandex::solomon::metabase::TLabelValuesResponse>();
        } else if (eventType == "TLabelNamesResponse") {
            return std::make_shared<yandex::solomon::metabase::TLabelNamesResponse>();
        } else if (eventType == "TUniqueLabelsResponse") {
            return std::make_shared<yandex::solomon::metabase::TUniqueLabelsResponse>();
        }
        Y_FAIL("Unknown event type: %s", eventType.c_str());
    }
};

class TLongTermSource: public IDataSource {
public:
    TLongTermSource(TActorRuntime& runtime, TLtsReplicas replicas)
        : Runtime_(runtime)
        , Replicas_(std::move(replicas))
    {
    }

    bool CanHandle(const TQuery&) const override {
        return true;
    }

    void Find(TFindQuery query, IResultHandlerPtr<TFindResult> handler, TSpanId traceCtx) const override {
        using TFindMergingActor = TMetaMergingActor<TFindMarshaller, TMetabaseEvents::TFindReq, TMetabaseEvents::TFindResp>;
        Runtime_.Register(new TFindMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void ResolveOne(TResolveOneQuery query, IResultHandlerPtr<TResolveOneResult> handler, TSpanId traceCtx) const override {
        using TResolveOneMergingActor = TMetaMergingActor<TResolveOneMarshaller, TMetabaseEvents::TResolveOneReq, TMetabaseEvents::TResolveOneResp>;
        Runtime_.Register(new TResolveOneMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void ResolveMany(TResolveManyQuery query, IResultHandlerPtr<TResolveManyResult> handler, TSpanId traceCtx) const override {
        using TResolveManyMergingActor = TMetaMergingActor<TResolveManyMarshaller, TMetabaseEvents::TResolveManyReq, TMetabaseEvents::TResolveManyResp>;
        Runtime_.Register(new TResolveManyMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void MetricNames(TMetricNamesQuery query, IResultHandlerPtr<TMetricNamesResult> handler, TSpanId traceCtx) const override {
        using TMetricNamesMergingActor = TMetaMergingActor<TMetricNamesMarshaller, TMetabaseEvents::TMetricNamesReq, TMetabaseEvents::TMetricNamesResp>;
        Runtime_.Register(new TMetricNamesMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void LabelKeys(TLabelKeysQuery query, IResultHandlerPtr<TLabelKeysResult> handler, TSpanId traceCtx) const override {
        using TLabelKeysMergingActor = TMetaMergingActor<TLabelKeysMarshaller, TMetabaseEvents::TLabelNamesReq, TMetabaseEvents::TLabelNamesResp>;
        Runtime_.Register(new TLabelKeysMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void LabelValues(TLabelValuesQuery query, IResultHandlerPtr<TLabelValuesResult> handler, TSpanId traceCtx) const override {
        using TLabelValuesMergingActor = TMetaMergingActor<TLabelValuesMarshaller, TMetabaseEvents::TLabelValuesReq, TMetabaseEvents::TLabelValuesResp>;
        Runtime_.Register(new TLabelValuesMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void UniqueLabels(TUniqueLabelsQuery query, IResultHandlerPtr<TUniqueLabelsResult> handler, TSpanId traceCtx) const override {
        using TUniqueLabelsMergingActor = TMetaMergingActor<TUniqueLabelsMarshaller, TMetabaseEvents::TUniqueLabelsReq, TMetabaseEvents::TUniqueLabelsResp>;
        Runtime_.Register(new TUniqueLabelsMergingActor{Replicas_, std::move(query), std::move(handler), std::move(traceCtx)});
    }

    void ReadOne(TReadOneQuery query, IResultHandlerPtr<TReadOneResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(ReadOneActor(Replicas_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    void ReadMany(TReadManyQuery query, IResultHandlerPtr<TReadManyResult> handler, TSpanId traceCtx) const override {
        Runtime_.Register(ReadManyActor(Replicas_, std::move(query), std::move(handler), std::move(traceCtx)));
    }

    using TInitFuture = NThreading::TFuture<THolder<TInitializationEvents::TInitialized>>;

    void WaitUntilInitialized() const override {
        TVector<TInitFuture> stockpileReplicasInitialized;
        TVector<TInitFuture> metabaseReplicasInitialized;

        for (auto replica: KnownReplicas) {
            stockpileReplicasInitialized.push_back(Runtime_.ActorSystem().Ask<TInitializationEvents::TInitialized>(
                    Replicas_[replica]->StockpileClusterId,
                    MakeHolder<TInitializationEvents::TSubscribe>()));

            metabaseReplicasInitialized.push_back(Runtime_.ActorSystem().Ask<TInitializationEvents::TInitialized>(
                    Replicas_[replica]->MetabaseClusterId,
                    MakeHolder<TInitializationEvents::TSubscribe>()));
        }
        auto success = NThreading::WaitAll(
                               NThreading::WaitAny(stockpileReplicasInitialized),
                               NThreading::WaitAny(metabaseReplicasInitialized))
                               .Wait(TDuration::Minutes(1));
        Y_VERIFY(success);
    }

private:
    TActorRuntime& Runtime_;
    TLtsReplicas Replicas_;
};

} // namespace

IDataSourcePtr LongTermSource(
        TActorRuntime& runtime,
        const TLongTermSourceConfig& config,
        const TClusterMap& clusterMap,
        const std::shared_ptr<NMonitoring::TMetricRegistry>& metrics,
        TString clientId)
{
    const auto& metabaseConfig = config.GetMetabaseConfig();
    const auto& stockpileConfig = config.GetStockpileConfig();
    auto metabaseRpc = CreateMetabaseRpc(metabaseConfig.GetClient(), *metrics, clientId);
    auto stockpileRpc = CreateStockpileRpc(stockpileConfig.GetClient(), *metrics, std::move(clientId));

    struct THosts {
        std::vector<TString> Metabase;
        std::vector<TString> Stockpile;
    };

    std::map<TClusterId, THosts> replicaHosts;
    for (const TString& address: metabaseRpc->Addresses()) {
        auto clusterId = clusterMap[address];
        Y_ENSURE(clusterId.has_value(), "cannot determine cluster by address \'" << address << '\'');
        replicaHosts[*clusterId].Metabase.push_back(address);
    }
    for (const TString& address: stockpileRpc->Addresses()) {
        auto clusterId = clusterMap[address];
        Y_ENSURE(clusterId.has_value(), "cannot determine cluster by address \'" << address << '\'');
        replicaHosts[*clusterId].Stockpile.push_back(address);
    }
    Y_ENSURE(replicaHosts.size() <= MAX_LTS_REPLICAS, "too many LTS replica clusters");

    // FIXME(ivanzhukov): get the tickDuration value from a config
    auto schedulerActor = CreateScheduler(runtime.ActorSystem().Timestamp(), TDuration::MilliSeconds(1));
    TActorId schedulerId = runtime.Register(schedulerActor.release());

    TLtsReplicas replicas;
    for (auto& [clusterId, hosts]: replicaHosts) {
        Y_ENSURE(!hosts.Metabase.empty(), "empty Metabase addresses list in cluster \'" << clusterId << '\'');
        Y_ENSURE(!hosts.Stockpile.empty(), "empty Stockpile addresses list in cluster \'" << clusterId << '\'');

        TActorId cacheId;
        if (metabaseConfig.HasCache()) {
            auto cache = CreateMessageCache(metabaseConfig.GetCache(), metrics, ELogComponent::MetaCache, "metabase", std::make_unique<TCacheResponses>());
            cacheId = runtime.Register(cache.release());
        }

        TString replicaStr = ToString(clusterId.Replica());
        TActorId metabaseId = runtime.Register(
                MetabaseCluster(
                        std::move(hosts.Metabase),
                        metabaseRpc,
                        CreateShardActorFactory(replicaStr, schedulerId),
                        metabaseConfig,
                        cacheId,
                        schedulerId).release(),
                TMailboxType::HTSwap,
                runtime.FindExecutorByName(metabaseConfig.GetClient().GetThreadPoolName()));

        TActorId stockpileId = runtime.Register(
                StockpileCluster(std::move(hosts.Stockpile), stockpileRpc, stockpileConfig).release(),
                TMailboxType::HTSwap,
                runtime.FindExecutorByName(stockpileConfig.GetClient().GetThreadPoolName()));

        replicas[clusterId.Replica()] = std::move(TLtsReplica{clusterId, metabaseId, stockpileId});
    }

    NSelfMon::RegisterPage(runtime.ActorSystem(), "/lts", "LTS Clusters", LtsSelfMonPage(replicas));
    return MakeIntrusive<TLongTermSource>(runtime, std::move(replicas));
}

} // namespace NSolomon::NDataProxy
