#pragma once

#include <solomon/services/dataproxy/lib/datasource/yasm/labels.h>
#include <solomon/services/dataproxy/lib/metric/metric.h>
#include <solomon/services/dataproxy/lib/selectors/shard_selector.h>

#include <solomon/libs/cpp/labels/known_keys.h>
#include <solomon/libs/cpp/selectors/selectors.h>
#include <solomon/libs/cpp/string_pool/string_pool.h>
#include <solomon/protos/model/selector.pb.h>

namespace NSolomon::NDataProxy {

/**
 * Converts string based TLabels into proto labels list (repeated yandex::solomon::model::TLabels).
 */
template <typename T>
void LabelsToProto(const TLabels<TString>& labels, T* labelsProto) {
    labelsProto->Reserve(labels.size());

    for (const auto& label: labels) {
        auto* labelProto = labelsProto->Add();
        labelProto->set_key(label.Key);
        labelProto->set_value(label.Value);
    }
}

/**
 * Converts proto labels (repeated yandex::solomon::model::TLabels) list into string based TLabels.
 */
template <typename T>
void LabelsFromProto(const T& labelsProto, TLabels<TString>* labels) {
    labels->reserve(labelsProto.size());

    for (const auto& labelProto: labelsProto) {
        labels->emplace_back(labelProto.key(), labelProto.value());
    }
}

/**
 * Converts id based TLabels into proto labels list (repeated yandex::solomon::model::TLabels).
 */
template <typename T>
void LabelsToProto(const NStringPool::TStringPool& strings, const TLabels<ui32>& labels, T* labelsProto) {
    labelsProto->Reserve(labels.size());

    for (const auto& label: labels) {
        auto* labelProto = labelsProto->Add();
        labelProto->set_key(TString{strings[label.Key]});
        labelProto->set_value(TString{strings[label.Value]});
    }
}

/**
 * Converts proto labels list (repeated yandex::solomon::model::TLabels) into id based TLabels.
 */
template <typename T>
void LabelsFromProto(const T& labelsProto, NStringPool::TStringPoolBuilder* strings, TLabels<ui32>* labels) {
    labels->reserve(labelsProto.size());

    for (const auto& labelProto: labelsProto) {
        ui32 key = strings->Put(labelProto.key());
        ui32 value = strings->Put(labelProto.value());
        labels->emplace_back(key, value);
    }
}

/**
 * Converts proto metric message (yandex::solomon::metabase::TMetric) to id based TMetricKey.
 */
template <typename T>
TMetricKey<ui32> MetricKeyFromProto(const T& metricProto, NStringPool::TStringPoolBuilder* strings) {
    TMetricKey<ui32> metricKey;
    metricKey.Name = strings->Put(metricProto.name());
    LabelsFromProto(metricProto.labels(), strings, &metricKey.Labels);
    return metricKey;
}

/**
 * Converts metcher type to protobuf enum value.
 */
inline yandex::solomon::model::MatchType MatcherTypeToProto(EMatcherType type, bool negative) {
    using yandex::solomon::model::MatchType;
    switch (type) {
        case EMatcherType::EXACT: return negative ? MatchType::NOT_EXACT : MatchType::EXACT;
        case EMatcherType::REGEX: return negative ? MatchType::NOT_REGEX : MatchType::REGEX;
        case EMatcherType::ANY: return MatchType::ANY;
        case EMatcherType::ABSENT: return MatchType::ABSENT;

        case EMatcherType::GLOB:
        case EMatcherType::MULTI:
            return negative ? MatchType::NOT_GLOB : MatchType::GLOB;
    }
}

/**
 * Converts selectors to protobuf selectors list (repeated yandex::solomon::model::TLabelSelector).
 */
template <typename T>
void SelectorsToProto(const TString& project, const TSelectors& selectors, T* selectorsProto) {
    using yandex::solomon::model::MatchType;

    selectorsProto->Reserve(selectors.size() + 1);

    {
        auto* sProto = selectorsProto->Add();
        sProto->set_key(TString{NLabels::LABEL_PROJECT});
        sProto->set_pattern(project);
        sProto->set_match_type(MatchType::EXACT);
    }

    for (const TSelector& s: selectors) {
        if (s.Key() == NLabels::LABEL_PROJECT) {
            // do not allow to override project selector
            continue;
        }

        auto* sProto = selectorsProto->Add();
        sProto->set_key(TString{s.Key()});
        sProto->set_pattern(TString{s.Pattern()});
        sProto->set_match_type(MatcherTypeToProto(s.Type(), s.Negative()));
    }
}

} // namespace NSolomon::NDataProxy
