#pragma once

#include "errors.h"
#include "lts.h"
#include "replica.h"

#include <solomon/services/dataproxy/lib/cluster_id/replica.h>
#include <solomon/services/dataproxy/lib/datasource/result_handler.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>

#include <solomon/libs/cpp/logging/logging.h>
#include <solomon/libs/cpp/trace/trace.h>

#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

namespace NSolomon::NDataProxy {

template <typename TMarshaller, typename TMetaReq, typename TMetaResp>
class TMetaMergingActor: public NActors::TActorBootstrapped<TMetaMergingActor<TMarshaller, TMetaReq, TMetaResp>> {
    using TQuery = typename TMarshaller::TQuery;
    using TResult = typename TMarshaller::TResult;

public:
    TMetaMergingActor(const TLtsReplicas& replicas, TQuery query, IResultHandlerPtr<TResult> handler, NTracing::TSpanId traceCtx)
        : Project_(query.Project)
        , Replicas_(replicas)
        , Deadline_(query.Deadline)
        , SoftDeadline_(query.SoftDeadline)
        , ForceReplicaRead_(query.ForceReplicaRead)
        , Marshaller_(std::move(query))
        , Handler_(std::move(handler))
        , TraceCtx_{std::move(traceCtx)}
    {
    }

    void Bootstrap() {
        this->Become(&TMetaMergingActor::StateFunc);

        auto msg = std::make_shared<typename TMetaReq::TProtoMsg>();
        Marshaller_.FillRequest(msg.get());
        auto shardSelector = Marshaller_.ShardSelector();

        for (auto replica: KnownReplicas) {
            const auto& maybeLtsReplica = Replicas_[replica];
            if (!maybeLtsReplica) {
                continue;
            }
            const TLtsReplica& ltsReplica = *maybeLtsReplica;
            if (!ForceReplicaRead_ || ltsReplica.Name() == ForceReplicaRead_) {
                auto req = std::make_unique<TMetaReq>();
                req->Deadline = Deadline_;
                req->ShardSelector = shardSelector;
                req->Message = msg;
                SendRequest(ltsReplica, std::move(req));
            }
        }

        if (AwaitingResponses_ == 0) {
            Handler_->OnError(
                    std::move(Project_),
                    EDataSourceStatus::BAD_REQUEST,
                    TStringBuilder{} << "invalid query, unknown force_replica_read = " << ForceReplicaRead_);
            this->PassAway();
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(TMetaResp, OnResponse);
            hFunc(TMetabaseEvents::TDone, OnDone);
            hFunc(TMetabaseEvents::TError, OnError);
            sFunc(NActors::TEvents::TEvWakeup, ReplySuccess);
        }
    }

    void SendRequest(const TLtsReplica& ltsReplica, std::unique_ptr<TMetaReq> request) {
        AwaitingResponses_++;
        auto span = TRACING_NEW_SPAN_START(TraceCtx_, "metabase-" << ltsReplica.ClusterId);
        this->Send(
                ltsReplica.MetabaseClusterId,
                request.release(),
                0,
                ToCookie(ltsReplica.ClusterId.Replica()),
                std::move(span));
    }

    void OnResponse(typename TMetaResp::TPtr ev) {
        MON_TRACE(MetaMerging, "got a response from Metabase: " << ev->Get()->Message->DebugString());

        auto replica = FromCookie(ev->Cookie);
        Marshaller_.AddResponse(replica, Replicas_[replica]->Dc(), *(ev->Get()->Message));
    }

    void OnDone(const TMetabaseEvents::TDone::TPtr& ev) {
        TRACING_SPAN_END_EV(ev);
        --AwaitingResponses_;
        auto replica = FromCookie(ev->Cookie);

        if (AwaitingResponses_ > 0) {
            if (!WereErrorsFromReplica_[replica]) {
                if (SoftDeadline_) {
                    // by schedule implementation, if soft deadline has already expired,
                    // event will be scheduled instantly
                    this->Schedule(*SoftDeadline_, new NActors::TEvents::TEvWakeup{});
                } else {
                    // wait response from second replica no more than 5s
                    // TODO: cancel replica request
                    this->Schedule(TDuration::Seconds(5), new NActors::TEvents::TEvWakeup{});
                }
            } // else wait for successful responses from other replicas

            return;
        }

        size_t numOfReplicas = 0;
        size_t numOfReplicasWithErrors = 0;
        for (const EReplica r: KnownReplicas) {
            if (Replicas_[r]) {
                numOfReplicas++;
                if (WereErrorsFromReplica_[r]) {
                    numOfReplicasWithErrors++;
                }
            }
        }

        if (numOfReplicasWithErrors == numOfReplicas) {
            ReplyError();
            return;
        }

        ReplySuccess();
    }

    void OnError(TMetabaseEvents::TError::TPtr& ev) {
        Errors_.emplace_back(ev->Release().Release());

        auto replica = FromCookie(ev->Cookie);
        WereErrorsFromReplica_[replica] = true;
    }

    void ReplySuccess() {
        TRACING_SPAN_END(TraceCtx_);
        Handler_->OnSuccess(Marshaller_.MakeResult());
        this->PassAway();
    }

    void ReplyError() {
        Y_VERIFY_DEBUG(!Errors_.empty());

        auto status = ToDataSourceStatus(Errors_[0]->RpcCode, Errors_[0]->MetabaseCode);
        if (Errors_.size() == 1) {
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(Project_), status, std::move(Errors_[0]->Message));
        } else {
            TStringBuilder message;
            for (auto& e: Errors_) {
                message << e->Message << ' ';
            }
            TRACING_SPAN_END(TraceCtx_);
            Handler_->OnError(std::move(Project_), status, std::move(message));
        }
        this->PassAway();
    }

private:
    static ui64 ToCookie(EReplica replica) {
        return static_cast<ui64>(replica);
    }

    static EReplica FromCookie(ui64 cookie) {
        return static_cast<EReplica>(cookie);
    }

private:
    TString Project_;
    const TLtsReplicas& Replicas_;
    TInstant Deadline_;
    std::optional<TInstant> SoftDeadline_;
    TString ForceReplicaRead_;
    TMarshaller Marshaller_;
    IResultHandlerPtr<TResult> Handler_;
    TReplicaMap<bool> WereErrorsFromReplica_{false};
    ui32 AwaitingResponses_{0};
    std::vector<std::unique_ptr<TMetabaseEvents::TError>> Errors_;
    NTracing::TSpanId TraceCtx_;
};

} // namespace NSolomon::NDataProxy
