#include "resolve_many_marshaller.h"
#include "marshaller_helpers.h"

#include <solomon/libs/cpp/proto_convert/metric_type.h>
#include <solomon/protos/metabase/grpc_resolve.pb.h>

namespace NSolomon::NDataProxy {

using yandex::solomon::metabase::ResolveManyRequest;
using yandex::solomon::metabase::ResolveManyResponse;

TResolveManyMarshaller::TResolveManyMarshaller(TResolveManyQuery query) noexcept
    : Query_(std::move(query))
{
}

TShardSelector TResolveManyMarshaller::ShardSelector() const {
    return TShardSelector::FromLabels(Query_.Project, Query_.Strings, Query_.CommonLabels);
}

void TResolveManyMarshaller::FillRequest(ResolveManyRequest* req) const {
    auto* commonLabels = req->mutable_commonlabels();
    commonLabels->Reserve(Query_.CommonLabels.size() + 1);

    auto* projectLabel = commonLabels->Add();
    projectLabel->set_key(TString{NLabels::LABEL_PROJECT});
    projectLabel->set_value(Query_.Project);

    for (const auto& label: Query_.CommonLabels) {
        TStringBuf key = Query_.Strings[label.Key];
        if (key == NLabels::LABEL_PROJECT) {
            // do not allow to override project
            continue;
        }
        auto* labelProto = commonLabels->Add();
        labelProto->set_key(TString{key});
        labelProto->set_value(TString{Query_.Strings[label.Value]});
    }

    auto* metrics = req->mutable_metrics();
    metrics->Reserve(Query_.MetricKeys.size());

    for (const auto& metricKey: Query_.MetricKeys) {
        auto* metric = metrics->Add();
        if (TStringBuf name = Query_.Strings[metricKey.Name]; !name.empty()) {
            metric->set_name(TString{name});
        }
        LabelsToProto(Query_.Strings, metricKey.Labels, metric->mutable_labels());
    }

    req->SetDeadlineMillis(Query_.Deadline.MilliSeconds());
}

void TResolveManyMarshaller::AddResponse(EReplica replica, EDc dc, const ResolveManyResponse& resp) {
    Dcs_[replica] = dc;
    if (Metrics_.empty()) {
        Metrics_.reserve(resp.metrics_size());
    }

    for (const auto& metric: resp.metrics()) {
        auto metricKey = MetricKeyFromProto(metric, &Strings_);
        auto [it, _] = Metrics_.emplace(std::move(metricKey), TMetricState{});

        TMetricState& state = it->second;
        state.Type = NSolomon::FromProto(metric.type());

        TStockpileId& stockpileId = state.StockpileIds[replica];
        stockpileId.ShardId = metric.metric_id().shard_id();
        stockpileId.LocalId = metric.metric_id().local_id();
    }
}

std::unique_ptr<TResolveManyResult> TResolveManyMarshaller::MakeResult() {
    auto result = std::make_unique<TResolveManyResult>();
    result->Strings = std::move(Strings_);

    result->Metrics.reserve(Metrics_.size());
    for (auto& [key, state]: Metrics_) {
        result->Metrics.emplace_back();
        auto& metric = result->Metrics.back();

        metric.Name = key.Name;
        metric.Labels = std::move(key.Labels); // TODO: move probably not working here
        metric.Type = state.Type;
        metric.StockpileIds = state.StockpileIds;
    }
    result->Dcs = Dcs_;
    return result;
}

} // namespace NSolomon::NDataProxy
