#include "lts_page.h"

#include <solomon/libs/cpp/selfmon/selfmon.h>
#include <solomon/libs/cpp/actors/poison/poisoner.h>

#include <library/cpp/actors/core/actor.h>
#include <library/cpp/actors/core/actor_bootstrapped.h>
#include <library/cpp/actors/core/hfunc.h>

namespace NSolomon::NDataProxy {
namespace {

class TClusterSelfMonPage: public NActors::TActor<TClusterSelfMonPage> {
public:
    explicit TClusterSelfMonPage(NActors::TActorId cluster) noexcept
        : NActors::TActor<TClusterSelfMonPage>{&TThis::StateFunc}
        , Cluster_{cluster}
    {
    }

    STFUNC(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            HFunc(NSelfMon::TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

    void OnRequest(NSelfMon::TEvPageDataReq::TPtr ev, const NActors::TActorContext& ctx) {
        ctx.Send(ev->Forward(Cluster_));
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr ev) {
        Send(ev->Sender, new NActors::TEvents::TEvPoisonTaken);
        PassAway();
    }

public:
    NActors::TActorId Cluster_;
};

class TLtsSelfMonPage: public NActors::TActorBootstrapped<TLtsSelfMonPage> {
public:
    explicit TLtsSelfMonPage(TLtsReplicas replicas) noexcept
        : Replicas_{std::move(replicas)}
    {
    }

    void Bootstrap() {
        Become(&TThis::StateFunc);

        for (auto replica: KnownReplicas) {
            auto& ltsReplica = Replicas_[replica];
            if (!ltsReplica) {
                continue;
            }
            TString clusterIdStr = ToString(ltsReplica->ClusterId);
            auto metabasePage = NSelfMon::RegisterPage(
                    *NActors::TActorContext::ActorSystem(),
                    "/lts/metabase/" + clusterIdStr,
                    "Metabase " + clusterIdStr,
                    std::make_unique<TClusterSelfMonPage>(ltsReplica->MetabaseClusterId),
                    true);

            auto stockpilePage = NSelfMon::RegisterPage(
                    *NActors::TActorContext::ActorSystem(),
                    "/lts/stockpile/" + clusterIdStr,
                    "Stockpile " + clusterIdStr,
                    std::make_unique<TClusterSelfMonPage>(ltsReplica->StockpileClusterId),
                    true);

            SubPages_.insert(metabasePage);
            SubPages_.insert(stockpilePage);
        }
    }

    STATEFN(StateFunc) {
        switch (ev->GetTypeRewrite()) {
            hFunc(NSelfMon::TEvPageDataReq, OnRequest)
            hFunc(NActors::TEvents::TEvPoison, OnPoison)
        }
    }

private:
    void OnRequest(NSelfMon::TEvPageDataReq::TPtr ev) {
        using namespace yandex::monitoring::selfmon;

        Page page;
        auto* component = page.mutable_component();
        auto* list = component->mutable_list();
        auto* links = list->mutable_reference();

        for (auto replica: KnownReplicas) {
            auto& ltsReplica = Replicas_[replica];
            if (!ltsReplica) {
                continue;
            }
            TString clusterIdStr = ToString(ltsReplica->ClusterId);

            auto* metabaseRef = links->add_values();
            metabaseRef->set_page("/lts/metabase/" + clusterIdStr);
            metabaseRef->set_title("Metabase " + clusterIdStr);

            auto* stockpileRef = links->add_values();
            stockpileRef->set_page("/lts/stockpile/" + clusterIdStr);
            stockpileRef->set_title("Stockpile " + clusterIdStr);
        }

        Send(ev->Sender, new NSelfMon::TEvPageDataResp{std::move(page)});
    }

    void OnPoison(NActors::TEvents::TEvPoison::TPtr ev) {
        PoisonAll(ev, SubPages_);
        PassAway();
    }

private:
    TLtsReplicas Replicas_;
    std::set<NActors::TActorId> SubPages_;
};

} // namespace

std::unique_ptr<NActors::IActor> LtsSelfMonPage(TLtsReplicas replicas) {
    return std::make_unique<TLtsSelfMonPage>(std::move(replicas));
}

} // namespace NSolomon::NDataProxy
