#include <solomon/services/dataproxy/lib/datasource/lts/data_read_actors.h>
#include <solomon/services/dataproxy/lib/datasource/ut_utils.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>
#include <solomon/services/dataproxy/lib/stockpile/events.h>
#include <solomon/services/dataproxy/lib/timeseries/protobuf.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NStockpile;
using namespace NSolomon::NDataProxy;
using namespace testing;
using namespace yandex::solomon::metabase;
using namespace yandex::solomon::stockpile;
using namespace yandex::solomon::model;

class TDataReadManyTest: public Test {
protected:
    void SetUp() override {
        Registry_ = std::make_shared<NMonitoring::TMetricRegistry>();

        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();

        for (auto replica: KnownReplicas) {
            Replicas_[replica] = TLtsReplica{
                    .ClusterId = TClusterId{EDc::Unknown, replica},
                    .MetabaseClusterId = EdgeId_,
                    .StockpileClusterId = EdgeId_};
        }
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    /**
     *  Dispatch `count` events other than logger events
     */
    void DispatchEvents(size_t count) {
        NActors::TDispatchOptions options;
        options.FinalEvents.emplace_back(
                [](IEventHandle& ev) {
                    return ev.GetTypeRewrite() != NActors::NLog::TEvLog::EventType;
                },
                count);

        ActorRuntime_->DispatchEvents(options);
    }

    template <typename TEvent>
    void Send(TActorId actorId, THolder<TEvent> event, size_t cookie) {
        ActorRuntime_->Send(actorId, EdgeId_, std::move(event), 0, cookie);
    }

    template <typename TEvent>
    typename TEvent::TPtr Receive() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(3));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
    TLtsReplicas Replicas_;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;
};

TReadManyQuery MakeQuery() {
    TReadManyQuery query;

    query.Project = "project";
    query.Time = {TInstant::Seconds(100), TInstant::Seconds(120)};
    query.Deadline = TInstant::Now() + TDuration::Seconds(15);
    query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();

    query.MaxTimeSeriesFormat = 39;

    auto downsampling = query.Operations.emplace_back().mutable_downsampling();
    downsampling->set_grid_millis(TDuration::Seconds(5).MilliSeconds());

    NStringPool::TStringPoolBuilder builder;
    auto& metric = query.ResolvedKeys->MetricKeys.emplace_back();
    metric.Name = builder.Put("");
    metric.Labels.emplace_back(builder.Put("signal"), builder.Put("signal"));
    metric.Labels.emplace_back(builder.Put("tag"), builder.Put("tag"));

    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("service"), builder.Put("service"));
    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("cluster"), builder.Put("cluster"));

    query.ResolvedKeys->Strings = builder.Build();
    return query;
}

TReadManyQuery MakeCrossShardQuery(size_t clusters) {
    TReadManyQuery query;

    query.Project = "project";
    query.Time = {TInstant::Seconds(100), TInstant::Seconds(120)};
    query.Deadline = TInstant::Now() + TDuration::Seconds(15);
    query.ResolvedKeys = std::make_unique<TReadManyQuery::TResolvedKeys>();

    query.MaxTimeSeriesFormat = 39;

    auto downsampling = query.Operations.emplace_back().mutable_downsampling();
    downsampling->set_grid_millis(TDuration::Seconds(5).MilliSeconds());

    NStringPool::TStringPoolBuilder builder;
    for (size_t i = 0; i < clusters; ++i) {
        auto& metric = query.ResolvedKeys->MetricKeys.emplace_back();
        metric.Name = builder.Put("");
        metric.Labels.emplace_back(builder.Put("signal"), builder.Put("signal"));
        metric.Labels.emplace_back(builder.Put("tag"), builder.Put("tag"));
        metric.Labels.emplace_back(builder.Put("cluster"), builder.Put("cluster" + ToString(i)));
    }
    query.ResolvedKeys->CommonLabels.emplace_back(builder.Put("service"), builder.Put("service"));

    query.ResolvedKeys->Strings = builder.Build();
    return query;
}

std::shared_ptr<ResolveManyResponse> MakeResolveManyResp(size_t metricsCount, size_t offset = 0) {
    auto response = std::make_shared<ResolveManyResponse>();
    response->set_status(EMetabaseStatusCode::OK);

    TVector<TString> keys{"project", "cluster", "service", "signal", "metricId"};

    for (size_t i = 1 + offset; i <= metricsCount + offset; ++i) {
        auto* m = response->add_metrics();
        m->set_type(MetricType::DGAUGE);
        m->set_name("");

        for (size_t j = 0; j + 1 < keys.size(); ++j) {
            auto* l = m->add_labels();
            l->set_key(keys[j]);
            l->set_value(keys[j]);
        }

        auto* l = m->add_labels();
        l->set_key(keys[keys.size() - 1]);
        l->set_value(ToString(i));


        m->mutable_metric_id()->set_shard_id(i);
        m->mutable_metric_id()->set_local_id(i);
    }

    return response;
}

TCompressedReadManyResponse MakeStockpileResponse(const TReadManyRequest& req) {
    TCompressedReadManyResponse resp;
    resp.set_status(EStockpileStatusCode::OK);

    auto metric = resp.add_metrics();
    metric->set_type(MetricType::DGAUGE);
    metric->set_shardid(req.shardid());
    metric->set_localid(req.localids(0));

    auto [readManyMetrics, readManyStrPool] = MakeMetricsWithTimeseries<double>(
            {{{
                      {"project", "project"},
                      {"service", "serivce"},
                      {"cluster", "cluster"},
                      {"tag", "tag"},
              },
              {0.0, 1.0, 2.0, 3.0},
              NMonitoring::EMetricType::GAUGE}},
            NZoom::NAccumulators::EAccumulatorType::Summ);

    auto* series = dynamic_cast<TCompressedTimeSeries*>(readManyMetrics[0].TimeSeries.get());

    ToProto(*series, metric->mutable_compressed(), EFormat::DELETED_SHARDS_39);

    return resp;
}

TEST_F(TDataReadManyTest, SimpleReadResolved) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    for (int i = 0; i < 2; ++i) {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, CrossShardReadResolved) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeCrossShardQuery(2), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    THashMap<TString, size_t> clusterToQueriesCount;
    for (int i = 0; i < 4; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();
        for (size_t i = 0; i < resolveReq->Get()->Message->CommonLabelsSize(); ++i) {
            auto label = resolveReq->Get()->Message->commonlabels(i);
            if (label.Getkey() == NLabels::LABEL_CLUSTER) {
                ++clusterToQueriesCount[label.Getvalue()];
            }
        }

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();

        // send shardId = 0 to 2 replicas for req1, then send shardId = 1 to 2 replicas for req2
        resolveResp->Message = MakeResolveManyResp(1, /* offset = */ i / 2);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    ASSERT_EQ(clusterToQueriesCount.size(), 2ul);
    for (const auto& [_, count]: clusterToQueriesCount) {
        ASSERT_EQ(count, 2ul);
    }

    for (int i = 0; i < 4; ++i) {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, EmptyMetabaseResponse) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, ErrorFromFirstStockpileReplica) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, ErrorFromSecondStockpileReplica) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    // TODO: check the content
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TDataReadManyTest, ErrorFromAllStockpileReplicas) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    for (int i = 0; i < 2; ++i) {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TDataReadManyTest, ErrorFromOneReplica) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TError>();
        resolveResp->MetabaseCode = EMetabaseStatusCode::RESOURCE_EXHAUSTED;

        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }
    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }


    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, ErrorFromOneReplica2) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TError>();
        resolveResp->MetabaseCode = EMetabaseStatusCode::RESOURCE_EXHAUSTED;

        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }
    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }


    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, ErrorFromMetabase) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TError>();
        resolveResp->MetabaseCode = EMetabaseStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, SomeMetricsInOneReplicaOnly) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();

        // 1 metric from R0, 2 metrics from R1
        resolveResp->Message = MakeResolveManyResp(i + 1); 

        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }


    // ok from R0
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // error from R1
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ok from R1
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, SomeMetricsInOneReplicaOnly2) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();

        // 1 metric from R0, 2 metrics from R1
        resolveResp->Message = MakeResolveManyResp(i + 1); 

        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }


    // error from R0
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // error from R1
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ok from R1
    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, MergeMetrics) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(2);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    /*
     *   This test implies that readReq->Cookie is value of counter
     *   that increases for each request starting from 1.
     */
     std::map<ui64, std::pair<ui64, EReplica>> idToCtx = {
         { 1, {1, EReplica::R0} },
         { 2, {2, EReplica::R0} },
         { 3, {1, EReplica::R1} },
         { 4, {2, EReplica::R1} },
     };


    for (int i = 0; i < 4; ++i) {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();
        ASSERT_EQ(idToCtx[readReq->Cookie].first, readReq->Get()->Message.shardid());

        if (readReq->Get()->Message.shardid() == 1) {
            // metric 1: error from R0, ok from R1
            if (idToCtx[readReq->Cookie].second == EReplica::R0) {
                auto readResp = MakeHolder<TStockpileEvents::TError>();
                readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
                Send(actorId, std::move(readResp), readReq->Cookie);
            } else {
                auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
                readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
                Send(actorId, std::move(readResp), readReq->Cookie);
            }
        } else {
            // metric 2: ok from R0, error from R1
            if (idToCtx[readReq->Cookie].second == EReplica::R0) {
                auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
                readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
                Send(actorId, std::move(readResp), readReq->Cookie);
            } else {
                auto readResp = MakeHolder<TStockpileEvents::TError>();
                readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
                Send(actorId, std::move(readResp), readReq->Cookie);
            }
        }
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TDataReadManyTest, DontAllowPartialResult) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(2);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }


    for (int i = 0; i < 4; ++i) {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        if (readReq->Get()->Message.shardid() == 1) {
            // metric 1: ok from R0, ok from R1
            auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
            readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
            Send(actorId, std::move(readResp), readReq->Cookie);
        } else {
            // metric 2: error from R0, error from R1
            auto readResp = MakeHolder<TStockpileEvents::TError>();
            readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
            Send(actorId, std::move(readResp), readReq->Cookie);
        }
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, DontWaitForSecondReplica) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    {
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(100));

    // Dispath timeout event
    DispatchEvents(2);

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, DontWaitForSecondReplica2) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    for (int i = 0; i < 2; ++i) {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(2);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    {
        // metric 1: error from R0
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();
        ASSERT_EQ(readReq->Get()->Message.shardid(), 1u);

        auto readResp = MakeHolder<TStockpileEvents::TError>();
        readResp->StockpileCode = EStockpileStatusCode::RESOURCE_EXHAUSTED;
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    {
        // metric 2: ok from R0
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();
        ASSERT_EQ(readReq->Get()->Message.shardid(), 2u);

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    auto readReqMetric1R1 = Receive<TStockpileEvents::TReadManyReq>();
    ASSERT_EQ(readReqMetric1R1->Get()->Message.shardid(), 1u);

    auto readReqMetric2R1 = Receive<TStockpileEvents::TReadManyReq>();
    ASSERT_EQ(readReqMetric2R1->Get()->Message.shardid(), 2u);

    // wait for reply from R1 for metric 1
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(100));

    // Dispath timeout event
    DispatchEvents(2);

    {
        // metric 1: ok from R1
        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReqMetric1R1->Get()->Message);
        Send(actorId, std::move(readResp), readReqMetric1R1->Cookie);
    }

    // do not wait for another reply for metric 1

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TDataReadManyTest, MetabaseSoftDeadline) {
    auto handlerMock = MakeIntrusive<TReadManyResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto actor = ReadManyActor(Replicas_, MakeQuery(), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor.release());
    ActorRuntime_->WaitForBootstrap();

    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();
    }

    {
        auto resolveReq = Receive<TMetabaseEvents::TResolveManyReq>();

        auto resolveResp = MakeHolder<TMetabaseEvents::TResolveManyResp>();
        resolveResp->Message = MakeResolveManyResp(1);
        Send(actorId, std::move(resolveResp), resolveReq->Cookie);

        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), resolveReq->Cookie);
    }

    // wait for reply from R0 for metric 1
    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(100));

    // Dispath wakeup event
    DispatchEvents(1);

    {
        // metric 1: error from R1
        auto readReq = Receive<TStockpileEvents::TReadManyReq>();
        ASSERT_EQ(readReq->Get()->Message.shardid(), 1u);

        auto readResp = MakeHolder<TStockpileEvents::TReadManyResp>();
        readResp->Message = MakeStockpileResponse(readReq->Get()->Message);
        Send(actorId, std::move(readResp), readReq->Cookie);
    }

    // ReadManyActor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}
