#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/find_marshaller.h>

#include <solomon/protos/metabase/grpc_find.pb.h>
#include <solomon/protos/model/selector.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::FindRequest;
using yandex::solomon::metabase::FindResponse;
using yandex::solomon::model::MatchType;

namespace {

struct TQueryTemplate: public TFindQuery {
    TQueryTemplate() {
        Project = "my-project";
        Selectors = ParseSelectors("{some = metric}");
        Deadline = TInstant::MilliSeconds(1234567890);
        Limit = 10;
    }
};

} // namespace

TEST(TFindMarshallerTest, FillRequest) {
    TFindMarshaller marshaller(TQueryTemplate{});

    FindRequest req;
    marshaller.FillRequest(&req);
    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "my-project"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::EXACT, "some", "metric"));
    EXPECT_EQ(1234567890u, req.GetDeadlineMillis());

    const auto& sliceOpts = req.GetSliceOptions();
    EXPECT_EQ(0u, sliceOpts.GetOffset());
    EXPECT_EQ(10, sliceOpts.GetLimit());
}

TEST(TFindMarshallerTest, EmptyResponse) {
    TFindMarshaller marshaller(TQueryTemplate{});

    FindResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(0u, result->Strings.Size());
    EXPECT_EQ(0u, result->Metrics.size());
    EXPECT_EQ(0u, result->TotalCount);
    EXPECT_TRUE(!result->Truncated);
}

TEST(TFindMarshallerTest, SingleResponse) {
    TFindMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<FindResponse>(R"(
        metrics {
            type: DGAUGE
            name: "my-name"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
        TotalCount: 100
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(5u, result->Strings.Size());
    EXPECT_EQ(1u, result->Metrics.size());
    EXPECT_EQ(100u, result->TotalCount);
    EXPECT_TRUE(!result->Truncated);

    auto& m = result->Metrics[0];
    auto resultStrings = result->Strings.Build();

    EXPECT_EQ(NMonitoring::EMetricType::GAUGE, m.Type);
    EXPECT_EQ("my-name", resultStrings[m.Name]);
    EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
    EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
    EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
    EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
}

TEST(TFindMarshallerTest, MultipleResponses_NoIntersecions) {
    TFindMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<FindResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 1, local_id: 11 }
        }
        TotalCount: 100
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Vla, ParseResp<FindResponse>(R"(
        metrics {
            type: RATE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 2, local_id: 21 }
        }
        TotalCount: 77
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(2u, result->Metrics.size());
    EXPECT_EQ(100u, result->TotalCount);
    EXPECT_TRUE(!result->Truncated);

    auto resultStrings = result->Strings.Build();

    for (const auto& m: result->Metrics) {
        if (m.Type == NMonitoring::EMetricType::RATE) {
            EXPECT_EQ("metric2", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
            EXPECT_EQ(2u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(21u, m.StockpileIds[EReplica::R1].LocalId);
        } else if (m.Type == NMonitoring::EMetricType::GAUGE) {
            EXPECT_EQ("metric1", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
            EXPECT_EQ(1u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(11u, m.StockpileIds[EReplica::R0].LocalId);
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TFindMarshallerTest, MultipleResponses_WithIntersecions) {
    TFindMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<FindResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 1, local_id: 11 }
        }
        TotalCount: 100
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Vla, ParseResp<FindResponse>(R"(
        metrics {
            type: RATE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 2, local_id: 21 }
        }
        TotalCount: 77
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(5u, result->Strings.Size());
    EXPECT_EQ(1u, result->Metrics.size());
    EXPECT_EQ(100u, result->TotalCount);
    EXPECT_TRUE(!result->Truncated);

    auto resultStrings = result->Strings.Build();

    auto& m = result->Metrics[0];
    EXPECT_EQ(NMonitoring::EMetricType::RATE, m.Type);
    EXPECT_EQ("metric1", resultStrings[m.Name]);

    EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
    EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
    EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
    EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);

    EXPECT_EQ(1u, m.StockpileIds[EReplica::R0].ShardId);
    EXPECT_EQ(11u, m.StockpileIds[EReplica::R0].LocalId);

    EXPECT_EQ(2u, m.StockpileIds[EReplica::R1].ShardId);
    EXPECT_EQ(21u, m.StockpileIds[EReplica::R1].LocalId);
}

TEST(TFindMarshallerTest, MultipleResponses_Limit) {
    TQueryTemplate query;
    query.Limit = 2;

    TFindMarshaller marshaller(query);
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<FindResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 1, local_id: 11 }
        }
        metrics {
            type: DGAUGE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 2, local_id: 21 }
        }
        metrics {
            type: DGAUGE
            name: "metric3"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 3, local_id: 31 }
        }
        TotalCount: 100
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Vla, ParseResp<FindResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 4, local_id: 41 }
        }
        metrics {
            type: DGAUGE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 5, local_id: 51 }
        }
        metrics {
            type: DGAUGE
            name: "metric3"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 6, local_id: 61 }
        }
        TotalCount: 77
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(7u, result->Strings.Size());
    EXPECT_EQ(2u, result->Metrics.size());
    EXPECT_EQ(100u, result->TotalCount);
    EXPECT_TRUE(!result->Truncated);

    EXPECT_EQ(query.Limit, result->Metrics.size());

    auto resultStrings = result->Strings.Build();

    for (const TMetric<ui32>& m: result->Metrics) {
        EXPECT_EQ(NMonitoring::EMetricType::GAUGE, m.Type);

        EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
        EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
        EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
        EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);

        if ("metric1" == resultStrings[m.Name]) {
            EXPECT_EQ(1u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(11u, m.StockpileIds[EReplica::R0].LocalId);

            EXPECT_EQ(4u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(41u, m.StockpileIds[EReplica::R1].LocalId);
        } else if ("metric2" == resultStrings[m.Name]) {
            EXPECT_EQ(2u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(21u, m.StockpileIds[EReplica::R0].LocalId);

            EXPECT_EQ(5u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(51u, m.StockpileIds[EReplica::R1].LocalId);
        } else if ("metric3" == resultStrings[m.Name]) {
            EXPECT_EQ(3u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(31u, m.StockpileIds[EReplica::R0].LocalId);

            EXPECT_EQ(6u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(61u, m.StockpileIds[EReplica::R1].LocalId);
        } else {
            FAIL() << "unexpected metric name: " << resultStrings[m.Name];
        }
    }
}
