#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/label_keys_marshaller.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <solomon/protos/metabase/grpc_label_names.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

#include <util/string/printf.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::TLabelNamesRequest;
using yandex::solomon::metabase::TLabelNamesResponse;
using yandex::solomon::model::MatchType;

namespace {

struct TQueryTemplate: public TLabelKeysQuery {
    TQueryTemplate() {
        Project = "my-project";
        Selectors = ParseSelectors("{some = metric}");
        Deadline = TInstant::MilliSeconds(1234567890);
    }
};

} // namespace

TEST(TLabelKeysMarshallerTest, FillRequest) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});

    TLabelNamesRequest req;
    marshaller.FillRequest(&req);

    EXPECT_EQ(2, req.selectors_size());
    EXPECT_TRUE(SelectorEq(req.selectors(0), MatchType::EXACT, "project", "my-project"));
    EXPECT_TRUE(SelectorEq(req.selectors(1), MatchType::EXACT, "some", "metric"));

    EXPECT_EQ(1234567890u, req.deadlinemillis());
}

TEST(TLabelKeysMarshallerTest, EmptyResponse) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});

    TLabelNamesResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(2u, result->Strings.Size());
    EXPECT_EQ(2u, result->Keys.size());

    auto resultStrings = result->Strings.Build();
    auto& keys = result->Keys;
    EXPECT_EQ("cluster", resultStrings[keys[0]]);
    EXPECT_EQ("service", resultStrings[keys[1]]);
}

TEST(TLabelKeysMarshallerTest, SingleResponse) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "zero", "one", "two", "three" ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(6u, result->Keys.size());

    auto resultStrings = result->Strings.Build();
    auto& keys = result->Keys;
    EXPECT_EQ("zero", resultStrings[keys[0]]);
    EXPECT_EQ("one", resultStrings[keys[1]]);
    EXPECT_EQ("two", resultStrings[keys[2]]);
    EXPECT_EQ("three", resultStrings[keys[3]]);
    EXPECT_EQ("cluster", resultStrings[keys[4]]);
    EXPECT_EQ("service", resultStrings[keys[5]]);
}

TEST(TLabelKeysMarshallerTest, NoClusterService) {
    TQueryTemplate query;
    query.Selectors = ParseSelectors("{some = metric, cluster=foo, service=bar}");
    TLabelKeysMarshaller marshaller(query);
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "zero", "one", "two", "three" ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(4u, result->Strings.Size());
    EXPECT_EQ(4u, result->Keys.size());

    auto& keys = result->Keys;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[keys[0]]);
    EXPECT_EQ("one", resultStrings[keys[1]]);
    EXPECT_EQ("two", resultStrings[keys[2]]);
    EXPECT_EQ("three", resultStrings[keys[3]]);
}

TEST(TLabelKeysMarshallerTest, MultipleResponses_NoIntersecions) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "zero", "one", "two" ]
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "three", "four", "five" ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(8u, result->Strings.Size());
    EXPECT_EQ(8u, result->Keys.size());

    auto& keys = result->Keys;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[keys[0]]);
    EXPECT_EQ("one", resultStrings[keys[1]]);
    EXPECT_EQ("two", resultStrings[keys[2]]);
    EXPECT_EQ("three", resultStrings[keys[3]]);
    EXPECT_EQ("four", resultStrings[keys[4]]);
    EXPECT_EQ("five", resultStrings[keys[5]]);
    EXPECT_EQ("cluster", resultStrings[keys[6]]);
    EXPECT_EQ("service", resultStrings[keys[7]]);
}

TEST(TLabelKeysMarshallerTest, MultipleResponses_WithIntersecions) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "zero", "one", "two" ]
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(R"(
        Names: [ "one", "two", "three" ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(6u, result->Keys.size());

    auto& keys = result->Keys;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[keys[0]]);
    EXPECT_EQ("one", resultStrings[keys[1]]);
    EXPECT_EQ("two", resultStrings[keys[2]]);
    EXPECT_EQ("three", resultStrings[keys[3]]);
    EXPECT_EQ("cluster", resultStrings[keys[4]]);
    EXPECT_EQ("service", resultStrings[keys[5]]);
}

// for viewing YASM data SOLOMON-7804
TEST(TLabelKeysMarshallerTest, ResponseWithSelf) {
    TLabelKeysMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelNamesResponse>(Sprintf(R"(
        Names: [ "%s", "%s", "two", "three" ]
    )", NYasm::AGGREGATED_MARKER.c_str(), NYasm::AGGREGATED_MARKER.c_str())));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(4u, result->Strings.Size());
    EXPECT_EQ(4u, result->Keys.size());

    auto& keys = result->Keys;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("two", resultStrings[keys[0]]);
    EXPECT_EQ("three", resultStrings[keys[1]]);
    EXPECT_EQ("cluster", resultStrings[keys[2]]);
    EXPECT_EQ("service", resultStrings[keys[3]]);
}
