#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/label_values_marshaller.h>

#include <solomon/libs/cpp/yasm/constants/labels.h>

#include <solomon/protos/metabase/grpc_label_values.pb.h>

#include <library/cpp/testing/gtest/gtest.h>
#include <library/cpp/protobuf/util/pb_io.h>

#include <util/generic/set.h>
#include <util/string/printf.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::TLabelValuesRequest;
using yandex::solomon::metabase::TLabelValuesResponse;
using yandex::solomon::model::MatchType;

namespace {

struct TQueryTemplate: public TLabelValuesQuery {
    TQueryTemplate() {
        Project = "my-project";
        Selectors = ParseSelectors("{some = metric}");
        Keys = {"key1", "key2", "key3"};
        TextFilter = "filter";
        Limit = 10;
        Deadline = TInstant::MilliSeconds(1234567890);
    }
};

} // namespace

void AssertValues(const NStringPool::TStringPool& strings, const TVector<ui32>& values, const TSet<TString>& expected) {
    EXPECT_EQ(expected.size(), values.size());

    TSet<TString> strValues;
    for (ui32 id: values) {
        strValues.insert(TString{strings[id]});
    }

    EXPECT_EQ(expected, strValues);
}

TEST(TLabelValuesMarshallerTest, FillRequest) {
    TLabelValuesMarshaller marshaller(TQueryTemplate{});

    TLabelValuesRequest req;
    marshaller.FillRequest(&req);

    EXPECT_EQ(2, req.selectors_size());
    EXPECT_TRUE(SelectorEq(req.selectors(0), MatchType::EXACT, "project", "my-project"));
    EXPECT_TRUE(SelectorEq(req.selectors(1), MatchType::EXACT, "some", "metric"));

    EXPECT_EQ(3, req.labels_size());
    EXPECT_EQ("key1", req.labels(0));
    EXPECT_EQ("key2", req.labels(1));
    EXPECT_EQ("key3", req.labels(2));

    EXPECT_EQ("filter", req.textsearch());
    EXPECT_EQ(10, req.limit());
    EXPECT_EQ(1234567890u, req.deadlinemillis());
}

TEST(TLabelValuesMarshallerTest, EmptyResponse) {
    TLabelValuesMarshaller marshaller(TQueryTemplate{});

    TLabelValuesResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(0u, result->Strings.Size());
    EXPECT_TRUE(result->Labels.empty());
    EXPECT_EQ(0u, result->MetricCount);
}

TEST(TLabelValuesMarshallerTest, SingleResponse) {
    TLabelValuesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelValuesResponse>(R"(
        Values: [
            {Name: "en", Values: ["zero", "one", "two"], Truncated: true, metric_count: 5 },
            {Name: "de", Values: ["nul", "ein", "zwei", "drei", "vier"], metric_count: 10 }
        ]
        metric_count: 10
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(10u, result->Strings.Size());
    EXPECT_EQ(2u, result->Labels.size());
    EXPECT_EQ(10u, result->MetricCount);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            AssertValues(resultStrings, l.Values, {"zero", "one", "two"});
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            AssertValues(resultStrings, l.Values, {"nul", "ein", "zwei", "drei", "vier"});
            EXPECT_EQ(10u, l.MetricCount);
            EXPECT_TRUE(!l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMarshallerTest, MultipleResponses_NoIntersecions) {
    TLabelValuesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelValuesResponse>(R"(
        Values: [
            {Name: "en", Values: ["zero", "one", "two"], Truncated: true, metric_count: 5 },
            {Name: "de", Values: ["nul", "ein", "zwei", "drei", "vier"], metric_count: 10 }
        ]
        metric_count: 10
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Sas, ParseResp<TLabelValuesResponse>(R"(
        Values: [
            {Name: "fr", Values: ["zéro", "une", "deux"], Truncated: true, metric_count: 3 },
            {Name: "es", Values: ["cero", "uno", "dos", "tres"], metric_count: 4 }
        ]
        metric_count: 7
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(19u, result->Strings.Size());
    EXPECT_EQ(4u, result->Labels.size());
    EXPECT_EQ(10u, result->MetricCount);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            AssertValues(resultStrings, l.Values, {"zero", "one", "two"});
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            AssertValues(resultStrings, l.Values, {"nul", "ein", "zwei", "drei", "vier"});
            EXPECT_EQ(10u, l.MetricCount);
            EXPECT_TRUE(!l.Truncated);
        } else if (resultStrings[l.Key] == "fr") {
            AssertValues(resultStrings, l.Values, {"zéro", "une", "deux"});
            EXPECT_EQ(3u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "es") {
            AssertValues(resultStrings, l.Values, {"cero", "uno", "dos", "tres"});
            EXPECT_EQ(4u, l.MetricCount);
            EXPECT_TRUE(!l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}

TEST(TLabelValuesMarshallerTest, MultipleResponses_WithIntersecions) {
    TLabelValuesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TLabelValuesResponse>(R"(
        Values: [
            {Name: "en", Values: ["zero", "two"], metric_count: 2 },
            {Name: "de", Values: ["ein", "drei"], metric_count: 3 }
        ]
        metric_count: 4
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Sas, ParseResp<TLabelValuesResponse>(R"(
        Values: [
            {Name: "en", Values: ["zero", "one", "two"], Truncated: true, metric_count: 5 },
            {Name: "de", Values: ["nul", "ein", "zwei"], metric_count: 10 }
        ]
        metric_count: 10
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(9u, result->Strings.Size());
    EXPECT_EQ(2u, result->Labels.size());
    EXPECT_EQ(10u, result->MetricCount);

    auto resultStrings = result->Strings.Build();

    for (auto& l: result->Labels) {
        if (resultStrings[l.Key] == "en") {
            AssertValues(resultStrings, l.Values, {"zero", "one", "two"});
            EXPECT_EQ(5u, l.MetricCount);
            EXPECT_TRUE(l.Truncated);
        } else if (resultStrings[l.Key] == "de") {
            AssertValues(resultStrings, l.Values, {"nul", "ein", "zwei", "drei"});
            EXPECT_EQ(10u, l.MetricCount);
            EXPECT_TRUE(!l.Truncated);
        } else {
            FAIL() << "unexpected label key";
        }
    }
}
