#include "selector_asserts.h"

#include <solomon/services/dataproxy/lib/datasource/lts/marshaller_helpers.h>

#include <solomon/protos/metabase/grpc_find.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::FindRequest;
using yandex::solomon::model::MatchType;

TEST(TMarshallerHelpersTest, EmptySelector) {
    FindRequest req;
    auto selectors = ParseSelectors("");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(1u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
}

TEST(TMarshallerHelpersTest, ExactSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host == stockpile-sas-000}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::EXACT, "host", "stockpile-sas-000"));
}

TEST(TMarshallerHelpersTest, NotExactSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host != stockpile-sas-000}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::NOT_EXACT, "host", "stockpile-sas-000"));
}

TEST(TMarshallerHelpersTest, GlobSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = stockpile-*-000}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::GLOB, "host", "stockpile-*-000"));
}

TEST(TMarshallerHelpersTest, NotGlobSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host != stockpile-*-000}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::NOT_GLOB, "host", "stockpile-*-000"));
}

TEST(TMarshallerHelpersTest, RegexSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host =~ stockpile-(sas|vla)-\\d+}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::REGEX, "host", "stockpile-(sas|vla)-\\d+"));
}

TEST(TMarshallerHelpersTest, NotRegexSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host !~ stockpile-(sas|vla)-\\d+}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::NOT_REGEX, "host", "stockpile-(sas|vla)-\\d+"));
}

TEST(TMarshallerHelpersTest, AnySelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = *}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::ANY, "host", "*"));
}

TEST(TMarshallerHelpersTest, AbsentSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = -}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::ABSENT, "host", "-"));
}

TEST(TMarshallerHelpersTest, MultiGlobSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = stockpile-sas-000, name = a|b}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(3u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::EXACT, "host", "stockpile-sas-000"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(2), MatchType::GLOB, "name", "a|b"));
}

TEST(TMarshallerHelpersTest, MultiNotGlobSelector) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = stockpile-sas-000, name != a|b}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(3u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::EXACT, "host", "stockpile-sas-000"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(2), MatchType::NOT_GLOB, "name", "a|b"));
}

TEST(TMarshallerHelpersTest, IgnoreProjectInSelectors) {
    FindRequest req;
    auto selectors = ParseSelectors("{host = stockpile-sas-000, project = secret_project}");
    SelectorsToProto("solomon", selectors, req.mutable_selectors());

    EXPECT_EQ(2u, req.SelectorsSize());
    EXPECT_TRUE(SelectorEq(req.GetSelectors(0), MatchType::EXACT, "project", "solomon"));
    EXPECT_TRUE(SelectorEq(req.GetSelectors(1), MatchType::EXACT, "host", "stockpile-sas-000"));
}

template <>
void Out<MatchType>(IOutputStream& out, MatchType type) {
    out << MatchType_Name(type);
}
