#include "google/protobuf/struct.pb.h"
#include "google/protobuf/wrappers.pb.h"
#include <solomon/services/dataproxy/lib/datasource/lts/meta_merging_actor.h>
#include <solomon/services/dataproxy/lib/datasource/query.h>
#include <solomon/services/dataproxy/lib/datasource/ut_utils.h>
#include <solomon/services/dataproxy/lib/metabase/events.h>

#include <solomon/libs/cpp/actors/test_runtime/actor_runtime.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NActors;
using namespace NSolomon;
using namespace NSolomon::NStockpile;
using namespace NSolomon::NDataProxy;
using namespace testing;
using namespace yandex::solomon::metabase;
using namespace yandex::solomon::stockpile;
using namespace yandex::solomon::model;

class TMetaMergingActorTest: public Test {
protected:
    void SetUp() override {
        Registry_ = std::make_shared<NMonitoring::TMetricRegistry>();

        ActorRuntime_ = TTestActorRuntime::CreateInited(1, false, true);
        ActorRuntime_->WaitForBootstrap();

        EdgeId_ = ActorRuntime_->AllocateEdgeActor();

        for (auto replica: KnownReplicas) {
            Replicas_[replica] = TLtsReplica{
                    .ClusterId = TClusterId{EDc::Unknown, replica},
                    .MetabaseClusterId = EdgeId_,
                    .StockpileClusterId = EdgeId_};
        }
    }

    void TearDown() override {
        ActorRuntime_.Reset();
    }

    /**
     *  Dispatch `count` events other than logger events
     */
    void DispatchEvents(size_t count) {
        NActors::TDispatchOptions options;
        options.FinalEvents.emplace_back(
                [](IEventHandle& ev) {
                    return ev.GetTypeRewrite() != NActors::NLog::TEvLog::EventType;
                },
                count);

        ActorRuntime_->DispatchEvents(options);
    }

    template <typename TEvent>
    void Send(TActorId actorId, THolder<TEvent> event, size_t cookie) {
        ActorRuntime_->Send(actorId, EdgeId_, std::move(event), 0, cookie);
    }

    template <typename TEvent>
    typename TEvent::TPtr Receive() {
        return ActorRuntime_->GrabEdgeEvent<TEvent>(EdgeId_, TDuration::Seconds(3));
    }

protected:
    THolder<TTestActorRuntime> ActorRuntime_;
    TActorId EdgeId_;
    TLtsReplicas Replicas_;
    std::shared_ptr<NMonitoring::TMetricRegistry> Registry_;
};

struct TTestQuery: public TQuery {
    TReplicaMap<bool> NeedResponseFromReplica;
};

struct TTestResult {};

struct TTestMetaReq: NActors::TEventLocal<TTestMetaReq, 0> {
    using TProtoMsg = google::protobuf::UInt64Value;

    TInstant Deadline;
    int ShardSelector = 1;
    std::shared_ptr<TProtoMsg> Message;
};

struct TTestMetaResp: NActors::TEventLocal<TTestMetaResp, 1> {
    std::shared_ptr<google::protobuf::UInt64Value> Message;
};

constexpr ui64 MESSAGE_VALUE = 17;

class TTestMarshaller {
public:
    using TQuery = TTestQuery;
    using TResult = TTestResult;

public:
    TTestMarshaller(TTestQuery query)
        : Query_{std::move(query)}
    {
    }

    int ShardSelector() {
        return 1;
    }

    void FillRequest(google::protobuf::UInt64Value* req) const {
        return req->set_value(MESSAGE_VALUE);
    }

    void AddResponse(EReplica replica, EDc, const google::protobuf::UInt64Value& value) {
        Y_VERIFY(Query_);
        Y_VERIFY(Query_->NeedResponseFromReplica[replica]);
        Y_VERIFY(value.value() == MESSAGE_VALUE);
        Query_->NeedResponseFromReplica[replica] = false;
    }

    std::unique_ptr<TTestResult> MakeResult() {
        Y_VERIFY(Query_);
        for (auto replica: KnownReplicas) {
            Y_VERIFY(!Query_->NeedResponseFromReplica[replica]);
        }
        return std::make_unique<TTestResult>();
    }

private:
    std::optional<TTestQuery> Query_;
};

TInstant DEADLINE = TInstant::Now() + TDuration::Seconds(10);

TTestQuery MakeQuery(bool needRespR0, bool needRespR1) {
    TTestQuery query;
    query.Deadline = DEADLINE;
    query.NeedResponseFromReplica[EReplica::R0] = needRespR0;
    query.NeedResponseFromReplica[EReplica::R1] = needRespR1;
    return query;
}

using TTestMetaMergingActor = TMetaMergingActor<TTestMarshaller, TTestMetaReq, TTestMetaResp>;

class TTestResultHandlerMock: public IResultHandler<TTestResult> {
public:
    MOCK_METHOD(void, OnSuccess, (std::unique_ptr<TTestResult>), (override));
    MOCK_METHOD(void, OnError, (TString&&, EDataSourceStatus, TString&&), (override));
};

TEST_F(TMetaMergingActorTest, Ok) {
    auto handlerMock = MakeIntrusive<TTestResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto* actor = new TTestMetaMergingActor(Replicas_, MakeQuery(true, true), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor);
    ActorRuntime_->WaitForBootstrap();

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto resp = MakeHolder<TTestMetaResp>();
        google::protobuf::UInt64Value value;
        value.set_value(MESSAGE_VALUE);
        resp->Message = std::make_shared<google::protobuf::UInt64Value>(value);
        Send(actorId, std::move(resp), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto resp = MakeHolder<TTestMetaResp>();
        google::protobuf::UInt64Value value;
        value.set_value(MESSAGE_VALUE);
        resp->Message = std::make_shared<google::protobuf::UInt64Value>(value);

        Send(actorId, std::move(resp), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }


    // Actor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}

TEST_F(TMetaMergingActorTest, OneError) {
    auto handlerMock = MakeIntrusive<TTestResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto* actor = new TTestMetaMergingActor(Replicas_, MakeQuery(true, false), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor);
    ActorRuntime_->WaitForBootstrap();

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto resp = MakeHolder<TTestMetaResp>();
        google::protobuf::UInt64Value value;
        value.set_value(MESSAGE_VALUE);
        resp->Message = std::make_shared<google::protobuf::UInt64Value>(value);
        Send(actorId, std::move(resp), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto err = MakeHolder<TMetabaseEvents::TError>();
        Send(actorId, std::move(err), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }


    // Actor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TMetaMergingActorTest, TwoErrors) {
    auto handlerMock = MakeIntrusive<TTestResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(1);
    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(0);

    auto* actor = new TTestMetaMergingActor(Replicas_, MakeQuery(false, false), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor);
    ActorRuntime_->WaitForBootstrap();

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto err = MakeHolder<TMetabaseEvents::TError>();
        Send(actorId, std::move(err), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto err = MakeHolder<TMetabaseEvents::TError>();
        Send(actorId, std::move(err), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }


    // Actor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}


TEST_F(TMetaMergingActorTest, DontWaitSecondReplica) {
    auto handlerMock = MakeIntrusive<TTestResultHandlerMock>();

    EXPECT_CALL(*handlerMock, OnSuccess(_)).Times(1);
    EXPECT_CALL(*handlerMock, OnError(_, _, _)).Times(0);

    auto* actor = new TTestMetaMergingActor(Replicas_, MakeQuery(true, false), handlerMock, NTracing::TSpanId{});

    auto actorId = ActorRuntime_->Register(actor);
    ActorRuntime_->WaitForBootstrap();

    {
        auto req = Receive<TTestMetaReq>();

        ASSERT_EQ(req->Get()->Message->value(), MESSAGE_VALUE);
        ASSERT_EQ(req->Get()->Deadline, DEADLINE);
        ASSERT_EQ(req->Get()->ShardSelector, 1);

        auto resp = MakeHolder<TTestMetaResp>();
        google::protobuf::UInt64Value value;
        value.set_value(MESSAGE_VALUE);
        resp->Message = std::make_shared<google::protobuf::UInt64Value>(value);
        Send(actorId, std::move(resp), req->Cookie);
        Send(actorId, MakeHolder<TMetabaseEvents::TDone>(), req->Cookie);
    }

    ActorRuntime_->AdvanceCurrentTime(TDuration::Seconds(10));

    // process TEvWakeup
    DispatchEvents(1);

    // Actor must die after calling the handler
    EXPECT_EQ(ActorRuntime_->FindActor(actorId), nullptr);
}
