#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/metric_names_marshaller.h>

#include <solomon/protos/metabase/grpc_metric_names.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::MetricNamesRequest;
using yandex::solomon::metabase::MetricNamesResponse;
using yandex::solomon::model::MatchType;

namespace {

struct TQueryTemplate: public TMetricNamesQuery {
    TQueryTemplate() {
        Project = "my-project";
        Selectors = ParseSelectors("{some = metric}");
        TextFilter = "filter";
        Deadline = TInstant::MilliSeconds(1234567890);
        Limit = 10;
    }
};

} // namespace
TEST(TMetricNamesMarshallerTest, FillRequest) {
    TMetricNamesMarshaller marshaller(TQueryTemplate{});

    MetricNamesRequest req;
    marshaller.FillRequest(&req);

    const auto& s = req.selectors();
    EXPECT_EQ(2, s.label_selectors_size());
    EXPECT_TRUE(SelectorEq(s.label_selectors(0), MatchType::EXACT, "project", "my-project"));
    EXPECT_TRUE(SelectorEq(s.label_selectors(1), MatchType::EXACT, "some", "metric"));

    EXPECT_EQ("filter", req.textsearch());
    EXPECT_EQ(10, req.limit());

    EXPECT_EQ(1234567890u, req.deadlinemillis());
}

TEST(TMetricNamesMarshallerTest, EmptyResponse) {
    TMetricNamesMarshaller marshaller(TQueryTemplate{});

    MetricNamesResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(0u, result->Strings.Size());
    EXPECT_TRUE(result->Names.empty());
    EXPECT_TRUE(!result->Truncated);
}

TEST(TMetricNamesMarshallerTest, SingleResponse) {
    TMetricNamesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<MetricNamesResponse>(R"(
        Names: [ "zero", "one", "two", "three" ]
        Truncated: false
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_TRUE(!result->Truncated);
    EXPECT_EQ(4u, result->Strings.Size());
    EXPECT_EQ(4u, result->Names.size());

    auto& names = result->Names;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[names[0]]);
    EXPECT_EQ("one", resultStrings[names[1]]);
    EXPECT_EQ("two", resultStrings[names[2]]);
    EXPECT_EQ("three", resultStrings[names[3]]);
}

TEST(TMetricNamesMarshallerTest, MultipleResponses_NoIntersecions) {
    TMetricNamesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<MetricNamesResponse>(R"(
        Names: [ "zero", "one", "two" ]
        Truncated: false
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<MetricNamesResponse>(R"(
        Names: [ "three", "four", "five" ]
        Truncated: true
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_TRUE(result->Truncated);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(6u, result->Names.size());

    auto& names = result->Names;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[names[0]]);
    EXPECT_EQ("one", resultStrings[names[1]]);
    EXPECT_EQ("two", resultStrings[names[2]]);
    EXPECT_EQ("three", resultStrings[names[3]]);
    EXPECT_EQ("four", resultStrings[names[4]]);
    EXPECT_EQ("five", resultStrings[names[5]]);
}

TEST(TMetricNamesMarshallerTest, MultipleResponses_WithIntersecions) {
    TMetricNamesMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<MetricNamesResponse>(R"(
        Names: [ "zero", "one", "two" ]
        Truncated: false
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<MetricNamesResponse>(R"(
        Names: [ "one", "two", "three" ]
        Truncated: false
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(4u, result->Strings.Size());
    EXPECT_EQ(4u, result->Names.size());

    auto& names = result->Names;
    auto resultStrings = result->Strings.Build();
    EXPECT_EQ("zero", resultStrings[names[0]]);
    EXPECT_EQ("one", resultStrings[names[1]]);
    EXPECT_EQ("two", resultStrings[names[2]]);
    EXPECT_EQ("three", resultStrings[names[3]]);
}
