#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/resolve_many_marshaller.h>

#include <solomon/protos/metabase/grpc_resolve.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::ResolveManyRequest;
using yandex::solomon::metabase::ResolveManyResponse;

namespace {

struct TQueryTemplate: public TResolveManyQuery {
    TQueryTemplate() {
        NStringPool::TStringPoolBuilder pool;
        auto s = [&pool](TStringBuf str) { return pool.Put(str); };

        Project = "monitoring";

        CommonLabels = {
                {s("project"), s("solomon")},
                {s("cluster"), s("production")},
                {s("service"), s("stockpile")},
        };

        MetricKeys = {
                {s("metric1"), { {s("key1"), s("value1")}, {s("key2"), s("value2")} }},
                {s("metric2"), { {s("key1"), s("value1")}, {s("key2"), s("value2")} }},
        };

        Strings = pool.Build();
        Deadline = TInstant::MilliSeconds(1234567890);
    }
};

} // namespace

TEST(TResolveManyMarshallerTest, FillRequest) {
    TResolveManyMarshaller marshaller(TQueryTemplate{});

    ResolveManyRequest req;
    marshaller.FillRequest(&req);
    EXPECT_EQ(3, req.commonlabels_size());
    EXPECT_TRUE(LabelEq(req.commonlabels(0), "project", "monitoring"));
    EXPECT_TRUE(LabelEq(req.commonlabels(1), "cluster", "production"));
    EXPECT_TRUE(LabelEq(req.commonlabels(2), "service", "stockpile"));
    EXPECT_EQ(1234567890u, req.GetDeadlineMillis());

    EXPECT_EQ(2, req.metrics_size());

    const auto& s1 = req.metrics(0);
    EXPECT_EQ("metric1", s1.name());
    EXPECT_EQ(2, s1.labels_size());
    EXPECT_TRUE(LabelEq(s1.labels(0), "key1", "value1"));
    EXPECT_TRUE(LabelEq(s1.labels(1), "key2", "value2"));

    const auto& s2 = req.metrics(1);
    EXPECT_EQ("metric2", s2.name());
    EXPECT_EQ(2, s2.labels_size());
    EXPECT_TRUE(LabelEq(s2.labels(0), "key1", "value1"));
    EXPECT_TRUE(LabelEq(s2.labels(1), "key2", "value2"));
}

TEST(TResolveManyMarshallerTest, EmptyResponse) {
    TResolveManyMarshaller marshaller(TQueryTemplate{});

    ResolveManyResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(0u, result->Strings.Size());
    EXPECT_TRUE(result->Metrics.empty());
}

TEST(TResolveManyMarshallerTest, SingleResponse) {
    TResolveManyMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveManyResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
        metrics {
            type: RATE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(2u, result->Metrics.size());

    auto resultStrings = result->Strings.Build();

    for (const auto& m: result->Metrics) {
        if (m.Type == NMonitoring::EMetricType::GAUGE) {
            EXPECT_EQ("metric1", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
        } else if (m.Type == NMonitoring::EMetricType::RATE) {
            EXPECT_EQ("metric2", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TResolveManyMarshallerTest, MultipleResponses_NoIntersecions) {
    TResolveManyMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveManyResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveManyResponse>(R"(
        metrics {
            type: RATE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(2u, result->Metrics.size());

    auto resultStrings = result->Strings.Build();

    for (const auto& m: result->Metrics) {
        if (m.Type == NMonitoring::EMetricType::GAUGE) {
            EXPECT_EQ("metric1", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
        } else if (m.Type == NMonitoring::EMetricType::RATE) {
            EXPECT_EQ("metric2", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}

TEST(TResolveManyMarshallerTest, MultipleResponses_WithIntersecions) {
    TResolveManyMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveManyResponse>(R"(
        metrics {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 1, local_id: 11 }
        }
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Vla, ParseResp<ResolveManyResponse>(R"(
        metrics {
            type: COUNTER
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 2, local_id: 21 }
        }
        metrics {
            type: RATE
            name: "metric2"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 3, local_id: 31 }
        }
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(6u, result->Strings.Size());
    EXPECT_EQ(2u, result->Metrics.size());

    auto resultStrings = result->Strings.Build();

    for (const auto& m: result->Metrics) {
        if (m.Type == NMonitoring::EMetricType::COUNTER) {
            EXPECT_EQ("metric1", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);

            EXPECT_EQ(1u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(11u, m.StockpileIds[EReplica::R0].LocalId);
            EXPECT_EQ(2u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(21u, m.StockpileIds[EReplica::R1].LocalId);
        } else if (m.Type == NMonitoring::EMetricType::RATE) {
            EXPECT_EQ("metric2", resultStrings[m.Name]);
            EXPECT_EQ("key1", resultStrings[m.Labels[0].Key]);
            EXPECT_EQ("value1", resultStrings[m.Labels[0].Value]);
            EXPECT_EQ("key2", resultStrings[m.Labels[1].Key]);
            EXPECT_EQ("value2", resultStrings[m.Labels[1].Value]);

            EXPECT_EQ(0u, m.StockpileIds[EReplica::R0].ShardId);
            EXPECT_EQ(0u, m.StockpileIds[EReplica::R0].LocalId);
            EXPECT_EQ(3u, m.StockpileIds[EReplica::R1].ShardId);
            EXPECT_EQ(31u, m.StockpileIds[EReplica::R1].LocalId);
        } else {
            FAIL() << "unexpected metric type";
        }
    }
}
