#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/resolve_one_marshaller.h>

#include <solomon/protos/metabase/grpc_resolve.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::ResolveOneRequest;
using yandex::solomon::metabase::ResolveOneResponse;

namespace {

struct TQueryTemplate: public TResolveOneQuery {
    TQueryTemplate() {
        Project = "my-project";
        Key = { "metric", { {"key1", "value1"}, {"key2", "value2"} } };
        Deadline = TInstant::MilliSeconds(1234567890);
    }
};

} // namespace

TEST(TResolveOneMarshallerTest, FillRequest) {
    TResolveOneMarshaller marshaller(TQueryTemplate{});

    ResolveOneRequest req;
    marshaller.FillRequest(&req);
    EXPECT_EQ("metric", req.name());
    EXPECT_EQ(2, req.labels_size());
    EXPECT_TRUE(LabelEq(req.labels(0), "key1", "value1"));
    EXPECT_TRUE(LabelEq(req.labels(1), "key2", "value2"));

    EXPECT_EQ(1234567890u, req.GetDeadlineMillis());
}

TEST(TResolveOneMarshallerTest, EmptyResponse) {
    TResolveOneMarshaller marshaller(TQueryTemplate{});

    ResolveOneResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_TRUE(result->Metric.Name.empty());
    EXPECT_TRUE(result->Metric.Labels.empty());
}

TEST(TResolveOneMarshallerTest, SingleResponse) {
    TResolveOneMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveOneResponse>(R"(
        metric {
            type: DGAUGE
            name: "my-name"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
        }
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);

    auto& m = result->Metric;
    EXPECT_EQ(NMonitoring::EMetricType::GAUGE, m.Type);
    EXPECT_EQ("my-name", m.Name);

    EXPECT_EQ(2u, m.Labels.size());

    EXPECT_EQ("key1", m.Labels[0].Key);
    EXPECT_EQ("value1", m.Labels[0].Value);

    EXPECT_EQ("key2", m.Labels[1].Key);
    EXPECT_EQ("value2", m.Labels[1].Value);
}

TEST(TResolveOneMarshallerTest, MultipleResponses) {
    TResolveOneMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<ResolveOneResponse>(R"(
        metric {
            type: DGAUGE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 1, local_id: 11 }
        }
    )"));

    marshaller.AddResponse(EReplica::R1, EDc::Vla, ParseResp<ResolveOneResponse>(R"(
        metric {
            type: RATE
            name: "metric1"
            labels { key: "key1", value: "value1" }
            labels { key: "key2", value: "value2" }
            metric_id { shard_id: 2, local_id: 21 }
        }
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);

    auto& m = result->Metric;
    EXPECT_EQ(NMonitoring::EMetricType::RATE, m.Type);
    EXPECT_EQ("metric1", m.Name);

    EXPECT_EQ(2u, m.Labels.size());

    EXPECT_EQ("key1", m.Labels[0].Key);
    EXPECT_EQ("value1", m.Labels[0].Value);

    EXPECT_EQ("key2", m.Labels[1].Key);
    EXPECT_EQ("value2", m.Labels[1].Value);

    EXPECT_EQ(1u, m.StockpileIds[EReplica::R0].ShardId);
    EXPECT_EQ(11u, m.StockpileIds[EReplica::R0].LocalId);
    EXPECT_EQ(2u, m.StockpileIds[EReplica::R1].ShardId);
    EXPECT_EQ(21u, m.StockpileIds[EReplica::R1].LocalId);
}
