#include "selector_asserts.h"
#include "parse_proto.h"

#include <solomon/services/dataproxy/lib/datasource/lts/unique_labels_marshaller.h>

#include <solomon/protos/metabase/grpc_label_unique.pb.h>

#include <library/cpp/testing/gtest/gtest.h>

using namespace NSolomon;
using namespace NDataProxy;
using yandex::solomon::metabase::TUniqueLabelsRequest;
using yandex::solomon::metabase::TUniqueLabelsResponse;
using yandex::solomon::model::MatchType;

namespace {

struct TQueryTemplate: public TUniqueLabelsQuery {
    TQueryTemplate() {
        Project = "my-project";
        Keys = {"key1", "key2", "key3"};
        Selectors = ParseSelectors("{some = metric}");
        Deadline = TInstant::MilliSeconds(1234567890);
    }
};

} // namespace

TEST(TUniqueLabelsMarshallerTest, FillRequest) {
    TUniqueLabelsMarshaller marshaller(TQueryTemplate{});

    TUniqueLabelsRequest req;
    marshaller.FillRequest(&req);

    EXPECT_EQ(2, req.selectors_size());
    EXPECT_TRUE(SelectorEq(req.selectors(0), MatchType::EXACT, "project", "my-project"));
    EXPECT_TRUE(SelectorEq(req.selectors(1), MatchType::EXACT, "some", "metric"));

    EXPECT_EQ(3, req.names_size());
    EXPECT_EQ("key1", req.names(0));
    EXPECT_EQ("key2", req.names(1));
    EXPECT_EQ("key3", req.names(2));

    EXPECT_EQ(1234567890u, req.deadlinemillis());
}

TEST(TUniqueLabelsMarshallerTest, EmptyResponse) {
    TUniqueLabelsMarshaller marshaller(TQueryTemplate{});

    TUniqueLabelsResponse resp;
    marshaller.AddResponse(EReplica::R0, EDc::Sas, resp);

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(0u, result->Strings.Size());
    EXPECT_TRUE(result->Labels.empty());
}

TEST(TUniqueLabelsMarshallerTest, SingleResponse) {
    TUniqueLabelsMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TUniqueLabelsResponse>(R"(
        LabelLists: [
            { labels: [
                {key: "key1", value: "value1"},
                {key: "key2", value: "value2"}
            ]},
            { labels: [
                {key: "key3", value: "value3"},
                {key: "key4", value: "value4"},
                {key: "key5", value: "value5"}
            ]}
        ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(10u, result->Strings.Size());
    EXPECT_EQ(2u, result->Labels.size());

    const auto& s = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_EQ("key1", s[labels[0].Key]);
            EXPECT_EQ("value1", s[labels[0].Value]);
            EXPECT_EQ("key2", s[labels[1].Key]);
            EXPECT_EQ("value2", s[labels[1].Value]);
        } else if (labels.size() == 3) {
            EXPECT_EQ("key3", s[labels[0].Key]);
            EXPECT_EQ("value3", s[labels[0].Value]);
            EXPECT_EQ("key4", s[labels[1].Key]);
            EXPECT_EQ("value4", s[labels[1].Value]);
            EXPECT_EQ("key5", s[labels[2].Key]);
            EXPECT_EQ("value5", s[labels[2].Value]);
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}

TEST(TUniqueLabelsMarshallerTest, MultipleResponses_NoIntersecions) {
    TUniqueLabelsMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TUniqueLabelsResponse>(R"(
        LabelLists: [
            { labels: [
                {key: "key1", value: "value1"},
                {key: "key2", value: "value2"}
            ]}
        ]
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TUniqueLabelsResponse>(R"(
        LabelLists: [
            { labels: [
                {key: "key3", value: "value3"},
                {key: "key4", value: "value4"},
                {key: "key5", value: "value5"}
            ]}
        ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(10u, result->Strings.Size());
    EXPECT_EQ(2u, result->Labels.size());

    const auto& s = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_EQ("key1", s[labels[0].Key]);
            EXPECT_EQ("value1", s[labels[0].Value]);
            EXPECT_EQ("key2", s[labels[1].Key]);
            EXPECT_EQ("value2", s[labels[1].Value]);
        } else if (labels.size() == 3) {
            EXPECT_EQ("key3", s[labels[0].Key]);
            EXPECT_EQ("value3", s[labels[0].Value]);
            EXPECT_EQ("key4", s[labels[1].Key]);
            EXPECT_EQ("value4", s[labels[1].Value]);
            EXPECT_EQ("key5", s[labels[2].Key]);
            EXPECT_EQ("value5", s[labels[2].Value]);
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}

TEST(TUniqueLabelsMarshallerTest, MultipleResponses_WithIntersecions) {
    TUniqueLabelsMarshaller marshaller(TQueryTemplate{});
    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TUniqueLabelsResponse>(R"(
        LabelLists: [
            { labels: [
                {key: "key1", value: "value1"},
                {key: "key2", value: "value2"}
            ]}
        ]
    )"));

    marshaller.AddResponse(EReplica::R0, EDc::Sas, ParseResp<TUniqueLabelsResponse>(R"(
        LabelLists: [
            { labels: [
                {key: "key1", value: "value1"},
                {key: "key2", value: "value2"}
            ]},
            { labels: [
                {key: "key3", value: "value3"},
                {key: "key4", value: "value4"},
                {key: "key5", value: "value5"}
            ]}
        ]
    )"));

    auto result = marshaller.MakeResult();
    ASSERT_TRUE(result);
    EXPECT_EQ(10u, result->Strings.Size());
    EXPECT_EQ(2u, result->Labels.size());

    const auto& s = result->Strings.Build();

    for (const auto& labels: result->Labels) {
        if (labels.size() == 2) {
            EXPECT_EQ("key1", s[labels[0].Key]);
            EXPECT_EQ("value1", s[labels[0].Value]);
            EXPECT_EQ("key2", s[labels[1].Key]);
            EXPECT_EQ("value2", s[labels[1].Value]);
        } else if (labels.size() == 3) {
            EXPECT_EQ("key3", s[labels[0].Key]);
            EXPECT_EQ("value3", s[labels[0].Value]);
            EXPECT_EQ("key4", s[labels[1].Key]);
            EXPECT_EQ("value4", s[labels[1].Value]);
            EXPECT_EQ("key5", s[labels[2].Key]);
            EXPECT_EQ("value5", s[labels[2].Value]);
        } else {
            FAIL() << "unexpected labels size";
        }
    }
}
