#include "merger.h"

namespace NSolomon::NDataProxy::NMerger {

TFindMerger::TFindMerger(size_t metricsLimit)
    : MetricsLimit_(metricsLimit)
{
}

void TFindMerger::AddResponse(TFindResult&& response, EStorageType type) {
    auto responseStrings = response.Strings.Build();

    for (const auto& item: response.Metrics) {
        TMetricKey<ui32> metricKey;
        metricKey.Labels.reserve(item.Labels.size());

        for (auto [key, value]: item.Labels) {
            ui32 k = Strings_.Put(responseStrings[key]);
            ui32 v = Strings_.Put(responseStrings[value]);
            metricKey.Labels.emplace_back(k, v);
        }

        metricKey.Name = Strings_.Put(responseStrings[item.Name]);

        std::sort(metricKey.Labels.begin(), metricKey.Labels.end(), [](auto lhs, auto rhs) {
            return lhs.Key < rhs.Key;
        });

        if (Metrics_.size() < MetricsLimit_) {
            auto [it, _] = Metrics_.emplace(std::move(metricKey), TMetricValue{item.Type});
            if (type == EStorageType::STS) {
                it->second.Type = item.Type;
            } else {
                it->second.LtsId = item.StockpileIds;
            }

        } else {
            Truncated_ = true;

            if (type == EStorageType::STS) {
                break;
            }

            if (auto it = Metrics_.find(metricKey); it != Metrics_.end()) {
                it->second.LtsId = item.StockpileIds;
            }
        }
    }

    Errors_.insert(Errors_.end(), response.Errors.begin(), response.Errors.end());

    if (type == EStorageType::STS) {
        STSTotalCount_ += response.TotalCount;
    } else {
        LTSTotalCount_ += response.TotalCount;
        LTSDcs_ = response.Dcs;
    }

    Truncated_ |= response.Truncated;
}

void TFindMerger::AddError(TDataSourceError error) {
    Errors_.emplace_back(std::move(error));
}

std::unique_ptr<TFindResult> TFindMerger::Finish() {
    auto r = std::make_unique<TFindResult>();
    r->Truncated = Truncated_;

    r->TotalCount = Max(STSTotalCount_, LTSTotalCount_, (ui32) Metrics_.size());
    r->Strings = std::move(Strings_);

    r->Metrics.reserve(Metrics_.size());

    r->Errors = std::move(Errors_);

    for (auto it = Metrics_.begin(); it != Metrics_.end(); ) {
        auto node = Metrics_.extract(it++);

        auto& metric = r->Metrics.emplace_back();
        metric.Type = node.mapped().Type;
        metric.Name = node.key().Name;
        metric.StockpileIds = node.mapped().LtsId;
        metric.Labels = std::move(node.key().Labels);
    }

    r->Dcs = LTSDcs_;

    return r;
}

} // NSolomon::NDataProxy::NMerger
